/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.data;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;

import de.jstacs.InstantiableFromParameterSet;
import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.WrongAlphabetException;
import de.jstacs.data.Alphabet.AlphabetParameterSet;
import de.jstacs.data.AlphabetContainerParameterSet.AlphabetArrayParameterSet;
import de.jstacs.data.AlphabetContainerParameterSet.SectionDefinedAlphabetParameterSet;
import de.jstacs.data.alphabets.ComplementableDiscreteAlphabet;
import de.jstacs.data.alphabets.ContinuousAlphabet;
import de.jstacs.data.alphabets.DiscreteAlphabet;
import de.jstacs.data.alphabets.DoubleSymbolException;
import de.jstacs.io.ArrayHandler;
import de.jstacs.io.ParameterSetParser;
import de.jstacs.io.XMLParser;
import de.jstacs.io.ParameterSetParser.NotInstantiableException;
import de.jstacs.parameters.InstanceParameterSet;
import de.jstacs.parameters.ParameterSet;
import de.jstacs.parameters.SimpleParameterSet;

/**
 * The container for some alphabets used in a sequence, sample, model or ... .
 * The container enables the user to have at each position a different alphabet
 * or at least not the same alphabet at all positions. This is impossible if you
 * use only instances of {@link Alphabet}. The container maps the given
 * {@link Alphabet} objects to the positions.
 * 
 * <br>
 * <br>
 * 
 * {@link AlphabetContainer} is immutable.
 * 
 * @author Jens Keilwagen
 * 
 * @see Alphabet
 */
public class AlphabetContainer implements Storable, InstantiableFromParameterSet, Comparable<AlphabetContainer> {

	private static final String XML_TAG = "AlphabetContainer";

	/**
	 * Possible types of {@link AlphabetContainer}s are discrete alphabets,
	 * continuous alphabets, or alphabets mixed of discrete and continuous
	 * symbols at different positions.
	 * 
	 * @author Jan Grau
	 */
	public enum AlphabetContainerType {
		/**
		 * The alphabet is discrete
		 */
		DISCRETE,
		/**
		 * The alphabet is continuous
		 */
		CONTINUOUS,
		/**
		 * The alphabet may be either continuous or discrete
		 */
		BOTH;

		static AlphabetContainerType determineType( Alphabet[] alphabets ) {
			boolean discrete = true;
			boolean continuous = true;
			for( int i = 0; i < alphabets.length; i++ ) {
				discrete &= ( alphabets[i] instanceof DiscreteAlphabet );
				continuous &= ( alphabets[i] instanceof ContinuousAlphabet );
				if( !( discrete || continuous ) ) {
					return BOTH;
				}
			}
			AlphabetContainerType type = null;
			if( discrete ) {
				type = DISCRETE;
			} else {
				type = AlphabetContainerType.CONTINUOUS;
			}
			return type;
		}
	}

	/**
	 * This method creates a new {@link AlphabetContainer} that uses as less as
	 * possible alphabets to describe the container. So, if possible, alphabets
	 * will be reused.
	 * 
	 * @param abc
	 *            the alphabets
	 * @param assignment
	 *            the assignment of the alphabets to the positions
	 * 
	 * @return an {@link AlphabetContainer} that uses as less as possible
	 *         alphabets
	 */
	public static AlphabetContainer getSimplifiedAlphabetContainer( Alphabet[] abc, int[] assignment ) {
		ArrayList<Alphabet> list = new ArrayList<Alphabet>( abc.length );
		Alphabet current;
		int[] assign = new int[assignment.length], abcAssign = new int[abc.length];
		Arrays.fill( abcAssign, -1 );
		for( int j, i = 0; i < assign.length; i++ ) {
			if( abcAssign[assignment[i]] < 0 ) {
				j = 0;
				current = abc[assignment[i]];
				while( j < list.size() && !list.get( j ).checkConsistency( current ) ) {
					j++;
				}
				abcAssign[i] = j;
				if( j == list.size() ) {
					list.add( current );
				}
			}
			assign[i] = abcAssign[assignment[i]];
		}

		return new AlphabetContainer( list.toArray( new Alphabet[0] ), assign );
	}

	/**
	 * This method may be used to construct a new {@link AlphabetContainer} by
	 * incorporating additional alphabets into an existing
	 * {@link AlphabetContainer}.
	 * 
	 * @param aC
	 *            the {@link AlphabetContainer} used as template for the
	 *            returned {@link AlphabetContainer}
	 * @param a
	 *            this {@link Alphabet} should be inserted
	 * @param useNewAlphabet
	 *            This array is used to define, which {@link Alphabet}s are used
	 *            for which positions. <br>
	 * <br>
	 *            <ul>
	 *            <li>the length of this array defines the length of
	 *            {@link Sequence}s the returned {@link AlphabetContainer} is
	 *            capable to handle
	 *            <li>for each position <code>false</code> forces to use the
	 *            alphabet given by the given {@link AlphabetContainer}
	 *            <li>for each position <code>true</code> forces to use the
	 *            given {@link Alphabet}
	 *            <ul>
	 *            <br>
	 *            <br>
	 *            Incorporation of the additional alphabet must be understood as
	 *            defining new additional positions to which the given alphabet
	 *            should be used. For example: let the given
	 *            {@link AlphabetContainer} contain three alphabets A0,A1,A2 (Ai
	 *            for position i) and have a possible length of three. Calling
	 *            this method using this {@link AlphabetContainer}, an
	 *            additional {@link Alphabet} A3 and an assignment-array of
	 *            <code>false</code>,<code>true</code>,<code>true</code>,
	 *            <code>false</code>,<code>false</code> returns a new
	 *            {@link AlphabetContainer} having a possible length of five and
	 *            using the following alphabets for those position:
	 *            A0,A3,A3,A1,A2. If the given {@link AlphabetContainer} has a
	 *            possible length not equal zero, then the assignment-array must
	 *            contain as many <code>false</code>-values as the length of the
	 *            given {@link AlphabetContainer}.
	 * 
	 * @return a new {@link AlphabetContainer} as described above
	 * 
	 * @throws IllegalArgumentException
	 *             if <code>useNewAlphabet</code> is <code>null</code> or has
	 *             length 0
	 */
	public static AlphabetContainer insertAlphabet( AlphabetContainer aC, Alphabet a, boolean[] useNewAlphabet ) throws IllegalArgumentException {

		if( useNewAlphabet == null || useNewAlphabet.length == 0 ) {
			throw new IllegalArgumentException( "given useNewAlphabet-array is null or has length 0" );
		}

		Alphabet[] tempAs = new Alphabet[aC.alphabet.length + 1];
		tempAs[tempAs.length - 1] = a;
		int[] tempAssign = new int[useNewAlphabet.length];

		int i;
		int pos1 = 0;

		if( aC.getPossibleLength() == 0 ) {

			for( i = 0; i < tempAssign.length; i++ ) {

				if( useNewAlphabet[i] ) {
					tempAssign[i] = 0;
				} else {
					tempAssign[i] = 1;
				}
			}

		} else {

			for( i = 0; i < tempAssign.length; i++ ) {

				if( useNewAlphabet[i] ) {
					tempAssign[i] = aC.index[pos1++];
				} else {
					tempAssign[i] = tempAs.length - 1;
				}
			}
		}
		return new AlphabetContainer( tempAs, tempAssign );
	}

	/**
	 * the underlying alphabets
	 */
	private Alphabet[] alphabet;

	private String delim;

	/**
	 * the assignment of the alphabets to the positions (in inhomogenous case)
	 */
	private int[] index;

	private AlphabetContainerParameterSet parameters;

	private double l;

	/**
	 * This constructor creates a simple {@link AlphabetContainer}. All
	 * positions use the same alphabet and therefore sequences of arbitrary
	 * length can be handled.
	 * 
	 * @param abc
	 *            the alphabet
	 */
	public AlphabetContainer( Alphabet abc ) {
		index = null;
		alphabet = new Alphabet[]{ abc };
		precompute();
	}

	/**
	 * This constructor creates an {@link AlphabetContainer} with different
	 * alphabets for each position. The assignment from the alphabets to the
	 * positions is given by the order in the alphabet array. This constructor
	 * should only be used if all alphabets are pairwise different.
	 * 
	 * @param abc
	 *            the alphabets
	 * 
	 * @see #AlphabetContainer(Alphabet[], int[])
	 */
	public AlphabetContainer( Alphabet[] abc ) {
		this( abc, null );
	}

	/**
	 * This constructor creates an new sparse {@link AlphabetContainer} based on
	 * given {@link AlphabetContainer}s.
	 * 
	 * @param cons
	 *            the {@link AlphabetContainer}s
	 * @param lengths
	 *            the corresponding lengths of each {@link AlphabetContainer}
	 *            that is used
	 * 
	 * @throws IllegalArgumentException
	 *             if the given length for an {@link AlphabetContainer} is not
	 *             possible
	 */
	public AlphabetContainer( AlphabetContainer[] cons, int[] lengths ) throws IllegalArgumentException {
		int j, i = 0, n = 0, k;
		for( ; i < lengths.length; i++ ) {
			j = cons[i].getPossibleLength();
			if( j != 0 && j != lengths[i] ) {
				throw new IllegalArgumentException( "The AlphabetContainer " + i
													+ " is not able to handle sequences of length "
													+ lengths[i]
													+ "." );
			} else {
				n += lengths[i];
			}
		}
		index = new int[n];
		int[] help;
		ArrayList<Alphabet> abcs = new ArrayList<Alphabet>( n );
		for( n = i = 0; i < lengths.length; i++ ) {
			help = new int[cons[i].alphabet.length];
			for( j = 0; j < help.length; j++ ) {
				k = 0;
				while( k < abcs.size() && !cons[i].alphabet[j].checkConsistency( abcs.get( k ) ) ) {
					k++;
				}
				if( k == abcs.size() ) {
					// no consistent alphabet was found
					// -> the alphabet need to be added
					abcs.add( cons[i].alphabet[j] );
				}
				// else // a consistent alphabet was found
				help[j] = k;
			}
			if( cons[i].index == null ) {
				for( j = 0; j < lengths[i]; j++ ) {
					index[n++] = help[0];
				}
			} else {
				for( j = 0; j < lengths[i]; j++ ) {
					index[n++] = help[cons[i].index[j]];
				}
			}
		}
		alphabet = abcs.toArray( new Alphabet[0] );
		if( alphabet.length == 1 ) {
			index = null;
		}
	}

	/**
	 * This constructor creates an {@link AlphabetContainer} that uses different
	 * alphabets. The alphabets can be used more than once. The assignment for
	 * the alphabets to the positions is given by the assignment-array.
	 * 
	 * @param abc
	 *            the alphabets
	 * @param assignment
	 *            the assignment-array
	 * 
	 * @throws IllegalArgumentException
	 *             if the assignment from the alphabets to the positions is not
	 *             correct
	 */
	public AlphabetContainer( Alphabet[] abc, int[] assignment ) throws IllegalArgumentException {
		if( abc.length == 1 ) {
			index = null;
		} else {
			int i = 0;
			if( assignment == null ) {
				index = new int[abc.length];
				while( i < index.length ) {
					index[i] = i++;
				}
			} else {
				index = new int[assignment.length];
				boolean[] used = new boolean[abc.length];
				while( i < index.length && 0 <= assignment[i] && assignment[i] < abc.length ) {
					used[assignment[i]] = true;
					index[i] = assignment[i++];
				}
				if( i < index.length ) {
					throw new IllegalArgumentException( "The assignment from the positions to the alphabets is corrupted at position " + i
														+ "." );
				}
				i = 1;
				while( i < used.length && ( used[0] &= used[i++] ) );
				if( !used[0] ) {
					throw new IllegalArgumentException( "The assignment from the positions to the alphabets is not surjective. (Not all alphabets are used.)" );
				}
			}
		}
		alphabet = abc.clone();
		precompute();
	}

	/**
	 * Creates a new {@link AlphabetContainer} from an
	 * {@link AlphabetContainerParameterSet} that contains all necessary
	 * parameters.
	 * 
	 * @param parameters
	 *            the parameters
	 * @throws IllegalArgumentException
	 *             is thrown if the {@link AlphabetContainerParameterSet} is not
	 *             cloneable
	 * @throws DoubleSymbolException
	 *             is thrown if the definitions within <code>parameters</code>
	 *             contains a symbol twice
	 * @throws NotInstantiableException
	 *             if an instance could not be created
	 */
	public AlphabetContainer( AlphabetContainerParameterSet parameters ) throws IllegalArgumentException, DoubleSymbolException,
																		NotInstantiableException {
		try {
			this.parameters = parameters.clone();
			ParameterSet alphSet = (ParameterSet)parameters.getParameterAt( 0 ).getValue();
			if( alphSet instanceof AlphabetParameterSet ) {
				// index = null;
				index = new int[1];
				alphabet = new Alphabet[]{ (Alphabet)ParameterSetParser.getInstanceFromParameterSet( (InstanceParameterSet)alphSet ) };
			} else if( alphSet instanceof AlphabetArrayParameterSet ) {
				index = new int[(Integer)alphSet.getParameterAt( 0 ).getValue()];
				alphabet = new Alphabet[alphSet.getNumberOfParameters() - 1];
				for( int i = 0; i < index.length; i++ ) {
					index[i] = i;
					ParameterSet par = (ParameterSet)alphSet.getParameterAt( i + 1 ).getValue();
					if( par instanceof SimpleParameterSet ) {
						par = (ParameterSet)par.getParameterAt( 0 ).getValue();
					}
					if( par instanceof AlphabetParameterSet ) {
						alphabet[i] = (Alphabet)ParameterSetParser.getInstanceFromParameterSet( (InstanceParameterSet)par );
					} else {
						throw new IllegalArgumentException( "The parameters of the alphabet at " + i + " are not given correctly" );
					}
				}
			} else if( alphSet instanceof SectionDefinedAlphabetParameterSet ) {
				if( !alphSet.hasDefaultOrIsSet() ) {
					throw new IllegalArgumentException( alphSet.getErrorMessage() );
				}
				index = new int[(Integer)alphSet.getParameterAt( 0 ).getValue()];
				alphabet = new Alphabet[alphSet.getNumberOfParameters() - 1];
				for( int i = 0; i < alphabet.length; i++ ) {
					ParameterSet singleAlph = (ParameterSet)alphSet.getParameterAt( i + 1 ).getValue();

					if( singleAlph.getParameterAt( 0 ).getValue() instanceof AlphabetParameterSet ) {
						alphabet[i] = (Alphabet)ParameterSetParser.getInstanceFromParameterSet( (InstanceParameterSet)singleAlph.getParameterAt( 0 )
								.getValue() );
					} else {
						throw new IllegalArgumentException( "Parameter for alphabet no. " + i
															+ " of unexpected type: "
															+ singleAlph.getClass()
															+ "." );
					}
					String section = (String)singleAlph.getParameterAt( 1 ).getValue();
					try {
						Iterator<Integer> posIt = SectionDefinedAlphabetParameterSet.parseSections( section ).iterator();
						while( posIt.hasNext() ) {
							index[posIt.next()] = i;
						}
					} catch ( Exception e ) {
						e.printStackTrace();
						throw new IllegalArgumentException( "Malformed sections for alphabet no. " + i + "." );
					}
				}
			} else {
				throw new IllegalArgumentException( "Wrong parameter type" );
			}
			precompute();
		} catch ( CloneNotSupportedException e ) {
			throw new IllegalArgumentException( e.getCause().getMessage() );
		}
	}

	/**
	 * Extracts the {@link AlphabetContainerParameterSet} from a
	 * {@link StringBuffer}.
	 * 
	 * @param xml
	 *            the XML stream
	 * 
	 * @throws NonParsableException
	 *             if the stream is not parsable
	 */
	public AlphabetContainer( StringBuffer xml ) throws NonParsableException {
		StringBuffer buf = XMLParser.extractForTag( xml, XML_TAG );
		alphabet = (Alphabet[])ArrayHandler.cast( XMLParser.extractStorableArrayForTag( buf, "Alphabets" ) );

		if( alphabet.length > 1 ) {
			index = XMLParser.extractIntArrayForTag( buf, "Assignment" );
			int i = 0;
			while( i < index.length && index[i] < alphabet.length ) {
				i++;
			}
			if( i < index.length ) {
				throw new NonParsableException( "The assignment from the positions to the alphabets is corrupted at position " + i + "." );
			}
		}
		precompute();
	}

	/**
	 * Checks whether two alphabets are consistent.
	 * 
	 * @param abc
	 *            the second alphabet
	 * 
	 * @return whether two alphabets are consistent
	 */
	public boolean checkConsistency( AlphabetContainer abc ) {
		return compareTo( abc ) == 0;
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo( AlphabetContainer abc ) {
		if( abc == this ) {
			return 0;
		}
		int i = 0, a1 = getPossibleLength(), a2 = abc.getPossibleLength();
		if( a1 == a2 ) {
			if( a1 == 0 ) {
				return alphabet[0].compareTo( abc.alphabet[0] );
			} else {
				boolean erg = true;
				// to be quicker
				byte notChecked = 0, consistent = 1, inconsistent = 2;
				byte[][] checked = new byte[alphabet.length][abc.alphabet.length];
				int current = 0;
				while( i < a2 && erg ) {
					if( checked[index[i]][abc.index[i]] == notChecked ) {
						current = alphabet[index[i]].compareTo( abc.alphabet[abc.index[i]] );
						checked[index[i]][abc.index[i]] = current == 0 ? consistent : inconsistent;
						erg &= checked[index[i]][abc.index[i]] == consistent;
					}
					i++;
				}
				return current;
			}
		} else {
			return a1 - a2;
		}
	}

	private void precompute() {
		int i = 0;
		while( i < alphabet.length && alphabet[i] instanceof DiscreteAlphabet
				&& ( (DiscreteAlphabet)alphabet[i] ).getMaximalSymbolLength() == 1 ) {
			i++;
		}
		delim = ( i == alphabet.length ) ? "" : " ";
		l = alphabet[0].length();
		for( i = 1; i < alphabet.length; i++ ) {
			l = Math.max( l, alphabet[i].length() );
		}
	}

	/**
	 * Returns the underlying alphabet of position <code>pos</code>. Please note
	 * that the alphabet is returned as reference, so take care of what you are
	 * doing with it!
	 * 
	 * @param pos
	 *            the position
	 * 
	 * @return the alphabet
	 */
	public Alphabet getAlphabetAt( int pos ) {
		if( alphabet.length == 1 ) {
			return alphabet[0];
		} else {
			return alphabet[index[pos]];
		}
	}

	/**
	 * Returns the length of the underlying alphabet of position
	 * <code>pos</code>.
	 * 
	 * @param pos
	 *            the position
	 * 
	 * @return the length of the underlying alphabet of position
	 *         <code>pos</code>
	 */
	public double getAlphabetLengthAt( int pos ) {
		if( alphabet.length == 1 ) {
			return alphabet[0].length();
		} else {
			return alphabet[index[pos]].length();
		}
	}

	/**
	 * Returns the encoded symbol <code>sym</code> for position <code>pos</code>
	 * .
	 * 
	 * @param pos
	 *            the position
	 * @param sym
	 *            the symbol
	 * 
	 * @return the encoded symbol
	 * 
	 * @throws WrongAlphabetException
	 *             if the symbol is not defined in the alphabet
	 */
	public double getCode( int pos, String sym ) throws WrongAlphabetException {
		if( isDiscreteAt( pos ) ) {
			return ( (DiscreteAlphabet)getAlphabetAt( pos ) ).getCode( sym );
		} else {
			double candidat = Double.parseDouble( sym );
			if( !( (ContinuousAlphabet)getAlphabetAt( pos ) ).isEncodedSymbol( candidat ) ) {
				throw new WrongAlphabetException();
			}
			return candidat;
		}
	}

	/**
	 * This method returns a container of alphabets e.g. for composite
	 * motifs/sequences.
	 * 
	 * @param start
	 *            the array of start indices
	 * @param length
	 *            the array of lengths
	 * 
	 * @return the container
	 * 
	 * @see AlphabetContainer#getSubContainer(int, int)
	 */
	public AlphabetContainer getCompositeContainer( int[] start, int[] length ) {
		if( alphabet.length == 1 ) {
			return this;
		} else {
			int i = 0, j, l = 0;
			for( ; i < length.length; i++ ) {
				l += length[i];
			}
			int[] ind = new int[l];
			int[] used = new int[alphabet.length];
			Arrays.fill( used, -1 );
			ArrayList<Alphabet> list = new ArrayList<Alphabet>();
			l = 0;
			for( i = 0; i < length.length; i++ ) {
				for( j = 0; j < length[i]; j++ ) {
					if( used[index[start[i] + j]] < 0 ) {
						used[index[start[i] + j]] = list.size();
						list.add( alphabet[index[start[i] + j]] );
					}
					ind[l++] = used[index[start[i] + j]];
				}
			}
			if( list.size() == 1 ) {
				return new AlphabetContainer( alphabet[start[0]] );
			} else {
				return new AlphabetContainer( list.toArray( new Alphabet[0] ), ind );
			}
		}
	}

	/* (non-Javadoc)
	 * @see de.jstacs.InstantiableFromParameterSet#getCurrentParameterSet()
	 */
	public AlphabetContainerParameterSet getCurrentParameterSet() throws Exception {
		if( parameters != null ) {
			return parameters.clone();
		} else {
			if( isSimple() ) {
				return new AlphabetContainerParameterSet( alphabet[0] );
			} else if( index.length == alphabet.length ) {
				return new AlphabetContainerParameterSet( alphabet );
			} else {
				return new AlphabetContainerParameterSet( alphabet, index );
			}
		}
	}

	/**
	 * Returns the delimiter that should be used (for writing e.g. a sequence).
	 * 
	 * @return the delimiter
	 */
	public String getDelim() {
		return delim;
	}

	/**
	 * Returns the maximal alphabet length of this container.
	 * 
	 * @return the maximal alphabet length of this container
	 */
	public double getMaximalAlphabetLength() {
		return l;
	}

	/**
	 * Returns the minimal value of the underlying alphabet of position
	 * <code>pos</code>.
	 * 
	 * @param pos
	 *            the position
	 * 
	 * @return the minimal value
	 */
	public double getMin( int pos ) {
		if( alphabet.length == 1 ) {
			return alphabet[0].getMin();
		} else {
			return alphabet[index[pos]].getMin();
		}
	}

	/**
	 * Returns the minimal alphabet length of this container.
	 * 
	 * @return the minimal alphabet length of this container
	 */
	public double getMinimalAlphabetLength() {
		double length = alphabet[0].length();
		for( int i = 1; i < alphabet.length; i++ ) {
			length = Math.min( length, alphabet[i].length() );
		}
		return length;
	}

	/**
	 * Returns the possible length for sequences using this container. If 0
	 * (zero) is returned, all lengths are possible.
	 * 
	 * @return the possible length using this container
	 */
	public int getPossibleLength() {
		return alphabet.length == 1 ? 0 : index.length;
	}

	/**
	 * This method returns a subcontainer for the positions starting at
	 * <code>start</code> and with length <code>length</code>. The method can be
	 * used for subsequences, ... .
	 * 
	 * @param start
	 *            the index of the start position
	 * @param length
	 *            the length
	 * 
	 * @return the subcontainer of alphabets
	 * 
	 * @see AlphabetContainer#getCompositeContainer(int[], int[])
	 */
	public AlphabetContainer getSubContainer( int start, int length ) {
		if( alphabet.length == 1 || ( start == 0 && length == getPossibleLength() ) ) {
			return this;
		} else {
			int[] ind = new int[length];
			int[] used = new int[alphabet.length];
			Arrays.fill( used, -1 );
			ArrayList<Alphabet> list = new ArrayList<Alphabet>();
			for( int i = 0; i < length; i++, start++ ) {
				if( used[index[start]] < 0 ) {
					used[index[start]] = list.size();
					list.add( alphabet[index[start]] );
				}
				ind[i] = used[index[start]];
			}
			if( list.size() == 1 ) {
				return new AlphabetContainer( alphabet[start] );
			} else {
				return new AlphabetContainer( list.toArray( new Alphabet[0] ), ind );
			}
		}
	}

	/**
	 * This method returns a {@link String} representation of <code>val</code>.
	 * 
	 * @param pos
	 *            the position
	 * @param val
	 *            the value
	 * 
	 * @return a {@link String} representation for <code>val</code> at position
	 *         <code>pos</code>
	 */
	public String getSymbol( int pos, double val ) {
		if( isDiscreteAt( pos ) ) {
			if( isSimple() ) {
				return ( (DiscreteAlphabet)alphabet[0] ).getSymbolAt( (int)val );
			} else {
				return ( (DiscreteAlphabet)alphabet[index[pos]] ).getSymbolAt( (int)val );
			}
		} else {
			return "" + val;
		}
	}

	/**
	 * If this method returns <code>true</code> all used alphabets ignore the
	 * case.
	 * 
	 * @return <code>true</code> if all used alphabets ignore the case
	 */
	public final boolean ignoresCase() {
		int i = 0;
		while( i < alphabet.length && ( alphabet[i] instanceof ContinuousAlphabet || ( (DiscreteAlphabet)alphabet[i] ).ignoresCase() ) ) {
			i++;
		}
		return alphabet.length == i;
	}

	/**
	 * If this method returns <code>true</code> all positions use discrete
	 * values.
	 * 
	 * @return <code>true</code> if all positions use discrete values
	 */
	public final boolean isDiscrete() {
		return getType() == AlphabetContainerType.DISCRETE;
	}

	/**
	 * Returns <code>true</code> if position <code>pos</code> is a discrete
	 * random variable.
	 * 
	 * @param pos
	 *            the position
	 * 
	 * @return <code>true</code> if position <code>pos</code> is a discrete
	 *         random variable
	 */
	public boolean isDiscreteAt( int pos ) {
		if( alphabet.length == 1 ) {
			return alphabet[0] instanceof DiscreteAlphabet;
		} else {
			return alphabet[index[pos]] instanceof DiscreteAlphabet;
		}
	}

	/**
	 * Returns <code>true</code> if <code>continuous</code> is a symbol of the
	 * alphabet used in position <code>pos</code>.
	 * 
	 * @param pos
	 *            the position
	 * @param continuous
	 *            the continuous value
	 * 
	 * @return <code>true</code> if <code>continuous</code> is a symbol of the
	 *         alphabet used in position <code>pos</code>
	 */
	public boolean isEncodedSymbol( int pos, double continuous ) {
		if( isDiscreteAt( pos ) ) {
			int discrete = toDiscrete( pos, continuous );
			return ( (DiscreteAlphabet)getAlphabetAt( pos ) ).isEncodedSymbol( discrete ) && ( discrete - continuous == 0d );
		} else {
			return ( (ContinuousAlphabet)getAlphabetAt( pos ) ).isEncodedSymbol( continuous );
		}
	}

	/**
	 * This method answers the question whether all random variables are defined
	 * over the same range, i.e. all positions use the same (fixed) alphabet.
	 * 
	 * @return whether all random variables are defined over the same range
	 */
	public final boolean isSimple() {
		return alphabet.length == 1;
	}

	/**
	 * This method helps to determine if the {@link AlphabetContainer} also
	 * computes the reverse complement of a sequence.
	 * 
	 * @return <code>true</code> if the {@link AlphabetContainer} also computes
	 *         the reverse complement of a sequence
	 */
	public final boolean isReverseComplementable() {
		return isSimple() && alphabet[0] instanceof ComplementableDiscreteAlphabet;
	}

	/**
	 * Returns the discrete value for <code>val</code> at a position
	 * <code>pos</code>.
	 * 
	 * @param pos
	 *            the position
	 * @param val
	 *            the value
	 * 
	 * @return a discrete value for <code>val</code> at position
	 *         <code>pos</code>
	 */
	public int toDiscrete( int pos, double val ) {
		if( isDiscreteAt( pos ) ) {
			return (int)val;
		} else {
			// TODO make it better (!?!, better discritisation)
			return (int)( val - getAlphabetAt( pos ).getMin() );
		}
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		String erg = "possible length: " + getPossibleLength() + "\n";
		erg += "alphabet : ";
		if( getPossibleLength() == 0 ) {
			erg += alphabet[0].toString();
		} else {
			for( int i = 0; i < getPossibleLength(); i++ ) {
				erg += "\n\t" + i + "\t" + getAlphabetAt( i ).toString();
			}
		}
		return erg + "\n";
	}

	/* (non-Javadoc)
	 * @see de.jstacs.Storable#toXML()
	 */
	public StringBuffer toXML() {
		StringBuffer xml = new StringBuffer( 300 + alphabet.length * 200 );
		XMLParser.appendStorableArrayWithTags( xml, alphabet, "Alphabets" );
		if( alphabet.length > 1 ) {
			XMLParser.appendIntArrayWithTags( xml, index, "Assignment" );
		}
		XMLParser.addTags( xml, XML_TAG );
		return xml;
	}

	/**
	 * Returns the type of this {@link AlphabetContainer}.
	 * 
	 * @return the type
	 */
	public final AlphabetContainerType getType() {
		return AlphabetContainerType.determineType( alphabet );
	}
}
