/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;

/**
 * This StringExtractor reads the Strings from a {@link File} as the user asks
 * for the String. Instances of this class are sparse since they do not store an
 * array of Strings internally.
 * 
 * @author Jens Keilwagen
 */
public class SparseStringExtractor extends AbstractStringExtractor {

	private BufferedReader reader;

	private String current, next;

	private StringBuffer help;

	/**
	 * A constructor that reads the lines from <code>file</code>.
	 * 
	 * @param file
	 *            the file to be read from
	 * 
	 * @throws IOException
	 *             is thrown if the file could not be read
	 * @throws FileNotFoundException
	 *             is thrown if the file could not be found
	 */
	public SparseStringExtractor( File file ) throws IOException, FileNotFoundException {
		this( file, USUALLY );
	}

	/**
	 * A constructor that reads the lines from <code>file</code> and ignores
	 * those starting with <code>ignore</code>.
	 * 
	 * @param file
	 *            the file to be read from
	 * @param ignore
	 *            the first character of lines that should be treated as
	 *            comments
	 * 
	 * @throws IOException
	 *             is thrown if the file could not be read
	 * @throws FileNotFoundException
	 *             is thrown if the file could not be found
	 */
	public SparseStringExtractor( File file, char ignore ) throws IOException, FileNotFoundException {
		this( file, ignore, file.getName() );
	}

	/**
	 * A constructor that reads the lines from <code>file</code> .
	 * 
	 * @param file
	 *            the file to be read from
	 * @param annotation
	 *            the annotation for the source
	 * 
	 * @throws IOException
	 *             is thrown if the file could not be read
	 * @throws FileNotFoundException
	 *             is thrown if the file could not be found
	 */
	public SparseStringExtractor( File file, String annotation ) throws IOException, FileNotFoundException {
		this( file, USUALLY, annotation );
	}

	/**
	 * A constructor that reads the lines from <code>file</code> and ignores
	 * those starting with <code>ignore</code>.
	 * 
	 * @param file
	 *            the file to be read from
	 * @param ignore
	 *            the first character of lines that should be treated as
	 *            comments
	 * @param annotation
	 *            the annotation for the source
	 * 
	 * @throws IOException
	 *             is thrown if the file could not be read
	 * @throws FileNotFoundException
	 *             is thrown if the file could not be found
	 */
	public SparseStringExtractor( File file, char ignore, String annotation ) throws IOException, FileNotFoundException {
		super( ignore );
		reader = new BufferedReader( new FileReader( file ) );
		help = new StringBuffer();
		next = getNext();
		this.annotation = annotation;
	}

	public boolean hasMoreElements() {
		return next != null;
	}

	public String nextElement() {
		current = next;
		next = getNext();
		return current;
	}

	private String getNext() {
		String res = null, line;
		try {
			while( (line = reader.readLine()) != null && ignorePattern.matcher( line ).matches() ){
				//System.out.println( "skip: " + line );
			}
			if( line == null || ignore != FASTA ) {
				res = line;
			} else {
				while( line != null && !ignorePattern.matcher( line ).matches()) {
					help.append( line );
					line = reader.readLine();
				}
				res = help.toString();
				help.delete( 0, help.length() );
			}
		} catch ( IOException e ) {
			throw new RuntimeException( "Forwarding IOException: " + e.getMessage() );
		}
		return res;
	}

	protected void finalize() throws Throwable {
		reader.close();
		super.finalize();
	}
}