/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */


package de.jstacs.io;

import de.jstacs.NonParsableException;
import de.jstacs.Storable;


/**
 * Class for parsing standard datatypes and arrays in and out of an XML file.
 * The methods with prefix <code>append</code> or <code>add</code> are for
 * encoding, while methods with prefix <code>extract</code> are for decoding.
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public class XMLParser
{	
	
	/**
	 * Extracts the code between equal start and end tags and returns the
	 * substring between the tags as new StringBuffer.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param tag
	 *            the tags between which a substring shall be taken (start tag should be equal to end tag)
	 * @return the substring between the tags as StringBuffer
	 * @throws NonParsableException
	 * 			  if the substring could not be parsed
	 */
	public static StringBuffer extractForTag( StringBuffer source, String tag ) throws NonParsableException
	{
		return extractForTag( source, tag, tag );
	}

	/**
	 * Returns the substring between start and end tag as StringBuffer.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the substring between start and end tag as StringBuffer
	 * @throws NonParsableException
	 * 			  if the substring could not be parsed
	 */
	public static StringBuffer extractForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		// System.out.println(startTag);
		if( source == null || source.length() == 0 )
		{
			return null;
		}
		int start = source.indexOf( "<" + startTag + ">" );
		if( start == -1 )
		{
			return null;
		}
		start += startTag.length() + 2;
		int end = start;
		int pos = start;
		int open = 1;
		while( ( pos = source.indexOf( endTag, pos + 1 ) ) > 0 )
		{
			if( source.charAt( pos - 1 ) == '/' && source.charAt( pos - 2 ) == '<' )
			{
				open--;
			}
			else if( source.charAt( pos - 1 ) == '<' )
			{
				open++;
			}
			// System.out.print(open+" ");
			if( open == 0 )
			{
				end = pos - 2;
				break;
			}
		}
		if( pos == -1 )
		{
			throw new NonParsableException( "Missing end tag for: " + endTag );
		}
		StringBuffer ex = new StringBuffer( source.substring( start, end ) );
		source.delete( start - startTag.length() - 2, end + endTag.length() + 3 );
		return ex;
	}

	/**
	 * Returns the value between equal start and end tags as boolean.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as boolean
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static boolean extractBooleanForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractBooleanForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as boolean.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as boolean
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static boolean extractBooleanForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		return Boolean.valueOf( extractForTag( source, startTag, endTag ).toString().trim() ).booleanValue();
	}
	
	/**
	 * Returns the value between equal start and end tags as byte.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as byte
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static byte extractByteForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractByteForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as byte.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as byte
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static byte extractByteForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		try
		{
			return Byte.parseByte( ex.toString() );
		} catch( NumberFormatException e )
		{
			throw new NonParsableException( "Could not parse \"" + startTag + "\" to int." );
		} catch( NullPointerException e )
		{
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
	}

	/**
	 * Returns the enumeration between equal start and end tags as Enum.
	 * 
	 * @param <T> 
	 * 			the type of the Enum objects
	 * @param source
	 * 			the source StringBuffer that should be decoded from XML
	 * @param tag
	 * 			the tags between which an enumeration shall be taken (start tag should be equal to end tag)
	 * @return the enumeration between the tags as Enum
	 * @throws NonParsableException
	 * 			if the enumeration could not be parsed
	 */
	public static <T extends Enum<T>> T extractEnumForTag( StringBuffer source, String tag ) throws NonParsableException
	{
		return extractEnumForTag( source, tag, tag );
	}
	
	
	/**
	 * Returns the enumeration between start and end tag as Enum.
	 * 
	 * @param <T>
	 * 			the type of the Enum objects
	 * @param source
	 * 			the source StringBuffer that should be decoded from XML
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * @return the enumeration between start and end tag as Enum
	 * @throws NonParsableException
	 * 			if the enumeration could not be parsed
	 */
	@SuppressWarnings("unchecked")
	public static <T extends Enum<T>> T extractEnumForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		if( ex == null )
		{
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
		String enumName = XMLParser.extractStringForTag( ex, "enumName" );
		T erg;
		try
		{
			erg = Enum.valueOf( (Class<T>) Class.forName( enumName ), XMLParser.extractStringForTag( ex, "name" ) );
        }
		catch( Exception e )
		{
			throw getNonParsableException( "problem at " + enumName + ": " + e.getClass().getSimpleName() + ": " + e.getCause().toString(), e );
		}
		return erg;
	}
	
	/**
	 * Returns the value between equal start and end tags as int.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as int
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static int extractIntForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractIntForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as int.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as int
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static int extractIntForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		try
		{
			return Integer.parseInt( ex.toString() );
		} catch( NumberFormatException e )
		{
			throw new NonParsableException( "Could not parse \"" + startTag + "\" to int." );
		} catch( NullPointerException e )
		{
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
	}

	/**
	 * Returns the value between equal start and end tags as long.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as long
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static long extractLongForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractLongForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as long.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as long
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static long extractLongForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		try
		{
			return Long.parseLong( ex.toString() );
		} catch( NumberFormatException e )
		{
			throw new NonParsableException( "Could not parse \"" + startTag + "\" to int." );
		} catch( NullPointerException e )
		{
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
	}

	/**
	 * Returns the value between equal start and end tags as double.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as double
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static double extractDoubleForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractDoubleForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as double.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as double
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static double extractDoubleForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		try
		{
			return Double.parseDouble( ex.toString() );
		} catch( NumberFormatException e )
		{
			throw new NonParsableException( "Could not parse \"" + startTag + "\" to double." );
		} catch( NullPointerException e )
		{
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
	}

	/**
	 * Returns the value between equal start and end tags as {@link Storable}.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as Storable
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static Storable extractStorableForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractStorableForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as {@link Storable}.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as Storable
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static Storable extractStorableForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		if( ex == null )
		{
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
		String className = XMLParser.extractStringForTag( ex, "className" );
		Storable erg;
		try
		{
			erg = (Storable) Class.forName( className ).getConstructor( new Class[] { StringBuffer.class } ).newInstance( ex );
        }
        catch( NoSuchMethodException e )
        {
        	throw getNonParsableException( "You must provide a constructor "+className+"(StringBuffer).", e );
        }
		catch( Exception e )
		{
			throw getNonParsableException( "problem at " + className + ": " + e.getClass().getSimpleName() + ": " + e.getCause().toString(), e );
		}
		return erg;
	}
	
	/**
	 * Returns the value between equal start and end tags as {@link Storable} or <code>null</code>.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param tag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as Storable or <code>null</code>
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */	
	public static Storable extractStorableOrNullForTag(StringBuffer source, String tag ) throws NonParsableException{
		return extractStorableOrNullForTag(source, tag, tag );
	}
	
	/**
	 * Returns the value between start and end tag as {@link Storable} or <code>null</code>.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as Storable or <code>null</code>
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static Storable extractStorableOrNullForTag(StringBuffer source, String startTag, String endTag ) throws NonParsableException{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		if( ex == null )
		{
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
		if(ex.toString().trim().equals( "null" )){
			return null;
		}
		String className = XMLParser.extractStringForTag( ex, "className" );

		Storable erg;
		try
		{
			erg = (Storable) Class.forName( className ).getConstructor( new Class[] { StringBuffer.class } ).newInstance( ex );
		}
		catch( NoSuchMethodException e )
		{
			throw getNonParsableException( "You must provide a constructor "+className+"(StringBuffer).", e );
		}
		catch( Exception e )
		{
			throw getNonParsableException( "problem at " + className + ": " + e.getClass().getSimpleName() + ": " + e.getCause().toString(), e );
		}
		return erg;
	}

	/**
	 * The method returns a NonParsableException with an own message for an Exception.
	 * 
	 * @param s
	 * 			the message of the NonParsableException
	 * @param e
	 * 			the Exception
	 * @return a NonParsableException with own message
	 */
	private static final NonParsableException getNonParsableException( String s, Exception e )
	{
		NonParsableException n = new NonParsableException( s );
		n.setStackTrace( e.getStackTrace() );
		return n;
	}
	
	/**
	 * Returns the value between equal start and end tags as a String.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a String
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static String extractStringForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractStringForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a String.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a String
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static String extractStringForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		if( ex == null )
		{
			throw new NonParsableException( "Could not find \"" + startTag + "\"." );
		}
		return ex.toString();
	}

	/**
	 * Returns the value between equal start and end tags as a boolean array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param tag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a boolean array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
    public static boolean[] extractBooleanArrayForTag(StringBuffer source, String tag) throws NonParsableException{
        return extractBooleanArrayForTag(source,tag,tag);
    }
    
    /**
	 * Returns the value between start and end tag as a boolean array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a boolean array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
    public static boolean[] extractBooleanArrayForTag(StringBuffer source, String startTag, String endTag) throws NonParsableException{
        StringBuffer ex = extractForTag(source,startTag,endTag);
        int l = extractIntForTag(ex,"length");
        boolean[] b = new boolean[l];
        for(int i=0;i<l;i++){
            b[i] = extractBooleanForTag(ex,  "pos val=\"" + i + "\"", "pos" );
        }
        return b;
    }
    
    /**
	 * Returns the value between equal start and end tags as a byte array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a byte array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static byte[] extractByteArrayForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractByteArrayForTag( source, startTag, startTag );
	}

	 /**
	 * Returns the value between start and end tag as a byte array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a byte array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static byte[] extractByteArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		byte[] array = new byte[l];
		for( int i = 0; i < l; i++ )
		{
			array[i] = extractByteForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}
    
	 /**
	 * Returns the value between equal start and end tags as an int array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as an int array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static int[] extractIntArrayForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractIntArrayForTag( source, startTag, startTag );
	}

	 /**
	 * Returns the value between start and end tag as an int array.
	 * 
	 * @param source
	 *            the source StringBuffer
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as an int array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static int[] extractIntArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		int[] array = new int[l];
		for( int i = 0; i < l; i++ )
		{
			array[i] = extractIntForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	 /**
	 * Returns the value between equal start and end tags as a double array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a double array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static double[] extractDoubleArrayForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractDoubleArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a double array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a double array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static double[] extractDoubleArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		double[] array = new double[l];
		for( int i = 0; i < array.length; i++ )
		{
			array[i] = extractDoubleForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a Storable array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a Storable array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static Storable[] extractStorableArrayForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractStorableArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a Storable array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a Storable array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static Storable[] extractStorableArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		Storable[] array = new Storable[l];
		for( int i = 0; i < array.length; i++ )
		{
			array[i] = extractStorableForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}
	
	/**
	 * Returns the value between equal start and end tags as a String array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a String array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static String[] extractStringArrayForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractStringArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a String array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a String array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static String[] extractStringArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		String[] array = new String[l];
		for( int i = 0; i < array.length; i++ )
		{
			array[i] = extractStringForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param tag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a two dimensional Storable array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
    public static Storable[][] extractStorable2ArrayForTag( StringBuffer source, String tag) throws NonParsableException{
        return extractStorable2ArrayForTag(source, tag, tag);
    }
    
    /**
	 * Returns the value between start and end tag as a two dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a two dimensional Storable array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
    public static Storable[][] extractStorable2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException{
        StringBuffer ex = extractForTag( source, startTag, endTag );
        int l = extractIntForTag( ex, "length" );
        Storable[][] array = new Storable[l][];
        for( int i = 0; i < l; i++ )
        {
            array[i] = extractStorableArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
        }
        return array;
    }
    
    /**
	 * Returns the value between equal start and end tags as a three dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param tag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a three dimensional Storable array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
    public static Storable[][][] extractStorable3ArrayForTag( StringBuffer source, String tag) throws NonParsableException{
        return extractStorable3ArrayForTag(source, tag, tag);
    }
    
    /**
	 * Returns the value between start and end tag as a three dimensional {@link Storable} array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a three dimensional Storable array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
    public static Storable[][][] extractStorable3ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException{
        StringBuffer ex = extractForTag( source, startTag, endTag );
        int l = extractIntForTag( ex, "length" );
        Storable[][][] array = new Storable[l][][];
        for( int i = 0; i < l; i++ )
        {
            array[i] = extractStorable2ArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
        }
        return array;
    }
    
    /**
	 * Returns the value between equal start and end tags as a two dimensional boolean array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param tag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a two dimensional boolean array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static boolean[][] extractBoolean2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException
	{
		return extractBoolean2ArrayForTag( source, tag, tag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional boolean array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a two dimensional boolean array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static boolean[][] extractBoolean2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		boolean[][] array = new boolean[extractIntForTag( ex, "length" )][];
		for( int i = 0; i < array.length; i++ )
		{
			array[i] = extractBooleanArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}
    
	/**
	 * Returns the value between equal start and end tags as a two dimensional byte array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a two dimensional byte array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static byte[][] extractByte2ArrayForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractByte2ArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional byte array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a two dimensional byte array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static byte[][] extractByte2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		byte[][] array = new byte[extractIntForTag( ex, "length" )][];
		for( int i = 0; i < array.length; i++ )
		{
			array[i] = extractByteArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}
    
	/**
	 * Returns the value between equal start and end tags as a two dimensional int array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a two dimensional int array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static int[][] extractInt2ArrayForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractInt2ArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional int array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a two dimensional int array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static int[][] extractInt2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int l = extractIntForTag( ex, "length" );
		int[][] array = new int[l][];
		for( int i = 0; i < l; i++ )
		{
			array[i] = extractIntArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional double array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a two dimensional double array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static double[][] extractDouble2ArrayForTag( StringBuffer source, String startTag ) throws NonParsableException
	{
		return extractDouble2ArrayForTag( source, startTag, startTag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional double array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a two dimensional double array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static double[][] extractDouble2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int i, l = extractIntForTag( ex, "length" );
		double[][] array = new double[l][];
		for( i = 0; i < l; i++ )
		{
			array[i] = extractDoubleArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Returns the value between equal start and end tags as a two dimensional String array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param tag
	 *            the tags between which a value shall be taken (start tag should be equal to end tag)
	 * @return the value between the tags as a two dimensional String array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static String[][] extractString2ArrayForTag( StringBuffer source, String tag ) throws NonParsableException
	{
		return extractString2ArrayForTag( source, tag, tag );
	}

	/**
	 * Returns the value between start and end tag as a two dimensional String array.
	 * 
	 * @param source
	 *            the source StringBuffer that should be decoded from XML
	 * @param startTag
	 *            the start tag
	 * @param endTag
	 *            the end tag
	 * @return the value between start and end tag as a two dimensional String array
	 * @throws NonParsableException
	 * 			  if the value could not be parsed
	 */
	public static String[][] extractString2ArrayForTag( StringBuffer source, String startTag, String endTag ) throws NonParsableException
	{
		StringBuffer ex = extractForTag( source, startTag, endTag );
		int i, l = extractIntForTag( ex, "length" );
		String[][] array = new String[l][];
		for( i = 0; i < l; i++ )
		{
			array[i] = extractStringArrayForTag( ex, "pos val=\"" + i + "\"", "pos" );
		}
		return array;
	}

	/**
	 * Frames the StringBuffer <code>source</code> with &quot;&lt; <code>tag</code>
	 * &gt;&quot; and &quot;&lt; <code>/tag</code> &gt;&quot;
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param tag
	 * 			the tags by which the StringBuffer should be framed (start tag should be equal to end tag)
	 */
	public static void addTags( StringBuffer source, String tag )
	{
		addTags( source, tag, tag );
	}

	/**
	 * Frames the StringBuffer <code>source</code> with &quot;&lt; <code>startTag</code>
	 * &gt;&quot; and &quot;&lt; <code>/endTag</code> &gt;&quot;
	 * 
	 * @param source
	 * 			 the source StringBuffer that should be encoded in XML
	 * @param startTag
	 * 			 the start tag
	 * @param endTag
	 * 			 the end tag
	 */
	public static void addTags( StringBuffer source, String startTag, String endTag )
	{
		source.insert( 0, "<" + startTag + ">\n" );
		source.insert( source.length(), "</" + endTag + ">\n" );
	}

	/**
	 * Appends a boolean value with equal tags to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>tag</code> &gt; <code>b</code> &lt;
	 * <code>/tag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param b
	 * 			the value that should be framed by the tags and appended to the source 
	 * @param tag
	 * 			the tags by which the value should be framed (start tag should be equal to end tag)
	 */
	public static void appendBooleanWithTags( StringBuffer source, boolean b, String tag )
	{
		appendBooleanWithTags( source, b, tag, tag );
	}

	/**
	 * Appends a boolean value with start and end tag to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>startTag</code> &gt; <code>b</code> &lt;
	 * <code>/endTag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param b
	 * 			the value that should be framed by start and end tag and appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendBooleanWithTags( StringBuffer source, boolean b, String startTag, String endTag )
	{
		source.append( "<" + startTag + ">" + b + "</" + endTag + ">\n" );
	}
	
	/**
	 * Appends a byte value with equal tags to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>tag</code> &gt; <code>b</code> &lt;
	 * <code>/tag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param b
	 * 			the value that should be framed by the tags and appended to the source 
	 * @param tag
	 * 			the tags by which the value should be framed (start tag should be equal to end tag)
	 */
	public static void appendByteWithTags( StringBuffer source, byte b, String tag )
	{
		appendIntWithTags( source, b, tag, tag );
	}
	
	/**
	 * Appends a byte value with start and end tag to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>startTag</code> &gt; <code>b</code> &lt;
	 * <code>/endTag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param b
	 * 			the value that should be framed by start and end tag and appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendByteWithTags( StringBuffer source, byte b, String startTag, String endTag )
	{
		source.append( "<" + startTag + ">" + b + "</" + endTag + ">\n" );
	}

	/**
	 * Appends an Enum object with equal tags to the StringBuffer <code>source</code>.
	 * 
	 * @param <T>
	 * 			the type of the Enum objects
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML	
	 * @param e
	 * 			the enumeration that should be framed by the tags and appended to the source 
	 * @param tag
	 * 			the tags by which the enumeration should be framed (start tag should be equal to end tag)
	 */
	public static <T extends Enum<T>> void appendEnumWithTags( StringBuffer source, Enum<T> e, String tag )
	{
		appendEnumWithTags( source, e, tag, tag );
	}
	
	/**
	 * Appends an Enum object with start and end tag to the StringBuffer <code>source</code>.
	 * 
	 * @param <T>
	 * 			the type of the Enum objects
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML	
	 * @param e
	 * 			the enumeration that should be framed by start and end tag and appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static <T extends Enum<T>> void appendEnumWithTags( StringBuffer source, Enum<T> e, String startTag, String endTag )
	{
		StringBuffer help = new StringBuffer( 1000 );
		help.append( "<" + startTag + ">\n");
		appendStringWithTags( help, e.getClass().getName(), "enumName" );
		appendStringWithTags( help, e.name(), "name" );
		help.append( "</" + endTag + ">\n" );
		source.append( help );
	}
	
	/**
	 * Appends an int value with equal tags to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>tag</code> &gt; <code>i</code> &lt;
	 * <code>/tag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the value that should be framed by the tags and appended to the source 
	 * @param tag
	 * 			the tags by which the value should be framed (start tag should be equal to end tag)
	 */
	public static void appendIntWithTags( StringBuffer source, int i, String tag )
	{
		appendIntWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an int value with start and end tag to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>startTag</code> &gt; <code>i</code> &lt;
	 * <code>/endTag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the value that should be framed by start and end tag and appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendIntWithTags( StringBuffer source, int i, String startTag, String endTag )
	{
		source.append( "<" + startTag + ">" + i + "</" + endTag + ">\n" );
	}
	
	/**
	 * Appends a long value with equal tags to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>tag</code> &gt; <code>i</code> &lt;
	 * <code>/tag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the value that should be framed by the tags and appended to the source 
	 * @param tag
	 * 			the tags by which the value should be framed (start tag should be equal to end tag)
	 */
	public static void appendLongWithTags( StringBuffer source, long i, String tag )
	{
		appendLongWithTags( source, i, tag, tag );
	}

	/**
	 * Appends a long value with start and end tag to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>startTag</code> &gt; <code>i</code> &lt;
	 * <code>/endTag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the value that should be framed by start and end tag and appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendLongWithTags( StringBuffer source, long i, String startTag, String endTag )
	{
		source.append( "<" + startTag + ">" + i + "</" + endTag + ">\n" );
	}

	/**
	 * Appends a double value with equal tags to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>tag</code> &gt; <code>d</code> &lt;
	 * <code>/tag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param d
	 * 			the value that should be framed by the tags and appended to the source 
	 * @param startTag
	 * 			the tags by which the value should be framed (start tag should be equal to end tag)
	 */
	public static void appendDoubleWithTags( StringBuffer source, double d, String startTag )
	{
		appendDoubleWithTags( source, d, startTag, startTag );
	}

	/**
	 * Appends a double value with start and end tag to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>startTag</code> &gt; <code>d</code> &lt;
	 * <code>/endTag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param d
	 * 			the value that should be framed by start and end tag and appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendDoubleWithTags( StringBuffer source, double d, String startTag, String endTag )
	{
		source.append( "<" + startTag + ">" + d + "</" + endTag + ">\n" );
	}

	/**
	 * Appends a {@link Storable} object with equal tags to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML	
	 * @param s
	 * 			the object that should be framed by the tags and appended to the source 
	 * @param startTag
	 * 			the tags by which the object should be framed (start tag should be equal to end tag)
	 */
	public static void appendStorableWithTags( StringBuffer source, Storable s, String startTag )
	{
		appendStorableWithTags( source, s, startTag, startTag );
	}

	/**
	 * Appends a {@link Storable} object with start and end tag to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML	
	 * @param s
	 * 			the object that should be framed by start and end tag and appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendStorableWithTags( StringBuffer source, Storable s, String startTag, String endTag )
	{
		StringBuffer help = new StringBuffer( 1000 );
		help.append( "<" + startTag + ">\n");
		appendStringWithTags( help, s.getClass().getName(), "className" );
		help.append( s.toXML() );
		help.append( "\t</" + endTag + ">\n" );
		source.append( help );
	}
	
	/**
	 * Appends a {@link Storable} object or &quot;null&quot; with equal tags to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML	
	 * @param s
	 * 			the object that should be framed by the tags and appended to the source, if <code>null</code> then &quot;null&quot; framed by the tags is appended to the source
	 * @param tag
	 * 			the tags by which the object or &quot;null&quot; should be framed (start tag should be equal to end tag)
	 */	
	public static void appendStorableOrNullWithTags( StringBuffer source, Storable s, String tag ) {
		appendStorableOrNullWithTags( source, s, tag, tag );
	}
	
	/**
	 * Appends a {@link Storable} object or &quot;null&quot; with start and end tag to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML	
	 * @param s
	 * 			the object that should be framed by start and end tag and appended to the source, if <code>null</code> then &quot;null&quot; framed by start and end tag is appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendStorableOrNullWithTags( StringBuffer source, Storable s, String startTag, String endTag ) {
		StringBuffer help = new StringBuffer( 1000 );
		help.append( "<" + startTag + ">\n");
		if(s != null){
			appendStringWithTags( help, s.getClass().getName(), "className" );
			help.append( s.toXML() );
		}else{
			help.append( "null" );
		}
		help.append( "\t</" + endTag + ">\n" );
		source.append( help );
	}

	/**
	 * Appends a String with equal tags to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>tag</code> &gt; <code>s</code> &lt;
	 * <code>/tag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the String that should be framed by the tags and appended to the source 
	 * @param startTag
	 * 			the tags by which the String should be framed (start tag should be equal to end tag)
	 */
	public static void appendStringWithTags( StringBuffer source, String s, String startTag )
	{
		appendStringWithTags( source, s, startTag, startTag );
	}

	/**
	 * Appends a String with start and end tag to the StringBuffer <code>source</code> in the following way: &quot;&lt; <code>startTag</code> &gt; <code>s</code> &lt;
	 * <code>/endTag</code> &gt;&quot; 
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the value that should be framed by start and end tag and appended to the source 
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendStringWithTags( StringBuffer source, String s, String startTag, String endTag )
	{
		source.append( "<" + startTag + ">" + s + "</" + endTag + ">\n" );
	}
	
	/**
	 * Encodes a byte array. The encoding is the following way: <br>
	 * &lt;length&gt;b.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt;b[0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;b[1]&lt;/pos&gt; <br>
	 * ... <br>
	 * It is necessary to include such an encoded array in tags or another
	 * array.
	 * 
	 * @param b 
	 * 			the array that should be encoded
	 * @return a StringBuffer representing the encoded array
	 * 
	 * @see XMLParser#addTags( StringBuffer source, String tag )
	 * @see XMLParser#addTags( StringBuffer source, String startTag, String
	 *      endTag )
	 */
	protected static StringBuffer ByteArrayWithTags( byte[] b )
	{
		int l = b.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			source.append( "\t" );
			appendIntWithTags( source, b[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Encodes an int array. The encoding is the following way: <br>
	 * &lt;length&gt;i.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt;i[0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;i[1]&lt;/pos&gt; <br>
	 * ... <br>
	 * It is necessary to include such an encoded array in tags or another
	 * array.
	 * 
	 * @param i 
	 * 			the array that should be encoded
	 * @return a StringBuffer representing the encoded array
	 * 
	 * @see XMLParser#addTags( StringBuffer source, String tag )
	 * @see XMLParser#addTags( StringBuffer source, String startTag, String
	 *      endTag )
	 */
	protected static StringBuffer IntArrayWithTags( int[] i )
	{
		int l = i.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			source.append( "\t" );
			appendIntWithTags( source, i[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Encodes a double array. The encoding is like in IntArrayWithTags.
	 * 
	 * @param d
	 * 			the array that should be encoded
	 * @return a StringBuffer representing the encoded array
	 * 
	 * @see XMLParser#IntArrayWithTags( int[] i )
	 */
	protected static StringBuffer DoubleArrayWithTags( double[] d )
	{
		int l = d.length;
		StringBuffer source = new StringBuffer( 25 + l * 40 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			source.append( "\t" );
			appendDoubleWithTags( source, d[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}
    
    /**
     * Encodes a boolean array. The encoding is like in IntArrayWithTags.
     * 
     * @param b
	 * 			the array that should be encoded
	 * @return a StringBuffer representing the encoded array
	 * 
     * @see XMLParser#IntArrayWithTags( int[] i )
     */
    protected static StringBuffer BooleanArrayWithTags( boolean[] b )
    {
        int l = b.length;
        StringBuffer source = new StringBuffer( 25 + l * 40 );
        source.append( "\t" );
        appendIntWithTags( source, l, "length" );
        for( int j = 0; j < l; j++ )
        {
            source.append( "\t" );
            appendBooleanWithTags( source, b[j], "pos val=\"" + j + "\"", "pos" );
        }
        return source;
    }

	/**
	 * Encodes a String array. The encoding is like in IntArrayWithTags.
	 * 
	 * @param s
	 * 			the array that should be encoded
	 * @return a StringBuffer representing the encoded array
	 * 
	 * @see XMLParser#IntArrayWithTags( int[] i )
	 */
	protected static StringBuffer StringArrayWithTags( String[] s )
	{
		int l = s.length;
		StringBuffer source = new StringBuffer( 25 + l * 30 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			source.append( "\t" );
			appendStringWithTags( source, s[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Encodes a Storable array.
	 * 
	 * @param s 
	 * 			the Storable object that should be encoded
	 * @return a StringBuffer representing the encoded Storable object
	 */
	protected static StringBuffer StorableArrayWithTags( Storable[] s )
	{
		int l = s.length;
		StringBuffer source = new StringBuffer( 25 + l * 1030 );
		source.append( "\t" );
		appendIntWithTags( source, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			source.append( "\t" );
			appendStorableWithTags( source, s[j], "pos val=\"" + j + "\"", "pos" );
		}
		return source;
	}

	/**
	 * Appends an encoded boolean array with equal tags to the StringBuffer <code>source</code>. The encoding is
	 * like in appendByteArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param b
	 * 			the array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByteArrayWithTags( StringBuffer source, byte[] b,
	 *      String startTag, String endTag )
	 */
    public static void appendBooleanArrayWithTags(StringBuffer source, boolean[] b, String tag){
        appendBooleanArrayWithTags(source,b,tag,tag);
    }
    
    /**
	 * Appends an encoded boolean array with start and end tag to the StringBuffer <code>source</code>. The encoding is
	 * like in appendByteArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param b
	 * 			the array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * 
	 * @see XMLParser#appendByteArrayWithTags( StringBuffer source, byte[] b,
	 *      String startTag, String endTag )
	 */
    public static void appendBooleanArrayWithTags(StringBuffer source, boolean[] b, String startTag, String endTag){
        StringBuffer help = BooleanArrayWithTags(b);
        addTags(help,startTag, endTag);
        source.append(help);
    }
    
    /**
	 * Appends an encoded byte array with equal tags to the StringBuffer <code>source</code>. The encoding is
	 * like in appendByteArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param b
	 * 			the array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByteArrayWithTags( StringBuffer source, byte[] b,
	 *      String startTag, String endTag )
	 */
	public static void appendByteArrayWithTags( StringBuffer source, byte[] b, String tag )
	{
		appendByteArrayWithTags( source, b, tag, tag );
	}
	
	/**
	 * Appends an encoded byte array with start and end tag to the StringBuffer <code>source</code>. The encoding
	 * is this way: <br>
	 * &lt; <code>startTag</code> &gt; &lt;length&gt;b.length&lt;/length&gt;
	 * <br>
	 * &lt;pos val=&quot;0&quot;&gt;b[0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;b[1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt; <code>/endTag</code> &gt;
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param b
	 * 			the array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendByteArrayWithTags( StringBuffer source, byte[] b, String startTag, String endTag )
	{
		StringBuffer help = ByteArrayWithTags( b );
		addTags( help, startTag, endTag );
		source.append( help );
	}
    
	/**
	 * Appends an encoded int array with equal tags to the StringBuffer <code>source</code>. The encoding is
	 * like in appendIntArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendIntArrayWithTags( StringBuffer source, int[] i,
	 *      String startTag, String endTag )
	 */
	public static void appendIntArrayWithTags( StringBuffer source, int[] i, String tag )
	{
		appendIntArrayWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an encoded int array with start and end tag to the StringBuffer <code>source</code>. The encoding
	 * is this way: <br>
	 * &lt; <code>startTag</code> &gt; &lt;length&gt;i.length&lt;/length&gt;
	 * <br>
	 * &lt;pos val=&quot;0&quot;&gt;i[0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;i[1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt; <code>/endTag</code> &gt;
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * 
	 */
	public static void appendIntArrayWithTags( StringBuffer source, int[] i, String startTag, String endTag )
	{
		StringBuffer help = IntArrayWithTags( i );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded double array with equal tags to the StringBuffer <code>source</code>. The
	 * encoding is like in appendIntArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param d
	 * 			the array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendIntArrayWithTags( StringBuffer source, int[] i,
	 *      String startTag, String endTag )
	 */
	public static void appendDoubleArrayWithTags( StringBuffer source, double[] d, String tag )
	{
		appendDoubleArrayWithTags( source, d, tag, tag );
	}

	/**
	 * Appends an encoded double array with start and end tag to the StringBuffer <code>source</code>. The
	 * encoding is like in appendIntArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param d
	 * 			the array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * 
	 * @see XMLParser#appendIntArrayWithTags( StringBuffer source, int[] i,
	 *      String startTag, String endTag )
	 */
	public static void appendDoubleArrayWithTags( StringBuffer source, double[] d, String startTag, String endTag )
	{
		StringBuffer help = DoubleArrayWithTags( d );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded Storable array with equal tags to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the Storable array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded Storable array should be framed (start tag should be equal to end tag)
	 * 
	 */
	public static void appendStorableArrayWithTags( StringBuffer source, Storable[] s, String tag )
	{
		appendStorableArrayWithTags( source, s, tag, tag );
	}

	/**
	 * Appends an encoded Storable array with start and end tag to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the Storable array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * 
	 */
	public static void appendStorableArrayWithTags( StringBuffer source, Storable[] s, String startTag, String endTag )
	{
		StringBuffer help = StorableArrayWithTags( s );
		addTags( help, startTag, endTag );
		source.append( help );
	}

	/**
	 * Appends an encoded String array with equal tags to the StringBuffer <code>source</code>. The
	 * encoding is like in appendIntArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendIntArrayWithTags( StringBuffer source, int[] i,
	 *      String startTag, String endTag )
	 */
	public static void appendStringArrayWithTags( StringBuffer source, String[] s, String tag )
	{
		appendStringArrayWithTags( source, s, tag, tag );
	}

	/**
	 * Appends an encoded String array with start and end tag to the StringBuffer <code>source</code>. The
	 * encoding is like in appendIntArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * 
	 * @see XMLParser#appendIntArrayWithTags( StringBuffer source, int[] i,
	 *      String startTag, String endTag )
	 */
	public static void appendStringArrayWithTags( StringBuffer source, String[] s, String startTag, String endTag )
	{
		StringBuffer help = StringArrayWithTags( s );
		addTags( help, startTag, endTag );
		source.append( help );
	}
	
	/**
	 * Appends an encoded two dimensional boolean array with equal tags to the StringBuffer <code>source</code>.
	 * The encoding is like in appendBoolean2ArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the two dimensional array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded two dimensional array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendBoolean2ArrayWithTags( StringBuffer source, boolean[][] i,
	 *      String startTag, String endTag )
	 */
	public static void appendBoolean2ArrayWithTags( StringBuffer source, boolean[][] i, String tag )
	{
		appendBoolean2ArrayWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional boolean array with start and end tag to the StringBuffer <code>source</code>.
	 * The encoding is the following way: <br>
	 * &lt;startTag&gt; &lt;length&gt;i.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt; &lt;length&gt;i[0].length&lt;/length&gt;
	 * <br>
	 * &lt;pos val=&quot;0&quot;&gt;i[0][0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;i[0][1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt; &lt;length&gt;i[1].length&lt;/length&gt;
	 * <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/endTag&gt;
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the two dimensional array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * 
	 */
	public static void appendBoolean2ArrayWithTags( StringBuffer source, boolean[][] i, String startTag, String endTag )
	{
		int l = i.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			help2 = BooleanArrayWithTags( i[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}
	
	/**
	 * Appends an encoded two dimensional byte array with equal tags to the StringBuffer <code>source</code>.
	 * The encoding is like in appendByte2ArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the two dimensional array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded two dimensional array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendByte2ArrayWithTags( StringBuffer source, byte[][] i,
	 *      String startTag, String endTag )
	 */
	public static void appendByte2ArrayWithTags( StringBuffer source, byte[][] i, String tag )
	{
		appendByte2ArrayWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional byte array with start and end tag to the StringBuffer <code>source</code>.
	 * The encoding is the following way: <br>
	 * &lt;startTag&gt; &lt;length&gt;i.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt; &lt;length&gt;i[0].length&lt;/length&gt;
	 * <br>
	 * &lt;pos val=&quot;0&quot;&gt;i[0][0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;i[0][1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt; &lt;length&gt;i[1].length&lt;/length&gt;
	 * <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/endTag&gt;
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the two dimensional array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendByte2ArrayWithTags( StringBuffer source, byte[][] i, String startTag, String endTag )
	{
		int l = i.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			help2 = ByteArrayWithTags( i[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional int array with equal tags to the StringBuffer <code>source</code>.
	 * The encoding is like in appendInt2ArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the two dimensional array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded two dimensional array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendInt2ArrayWithTags( StringBuffer source, int[][] i,
	 *      String startTag, String endTag )
	 */
	public static void appendInt2ArrayWithTags( StringBuffer source, int[][] i, String tag )
	{
		appendInt2ArrayWithTags( source, i, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional int array with start and end tag to the StringBuffer <code>source</code>.
	 * The encoding is the following way: <br>
	 * &lt;startTag&gt; &lt;length&gt;i.length&lt;/length&gt; <br>
	 * &lt;pos val=&quot;0&quot;&gt; &lt;length&gt;i[0].length&lt;/length&gt;
	 * <br>
	 * &lt;pos val=&quot;0&quot;&gt;i[0][0]&lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt;i[0][1]&lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * &lt;pos val=&quot;1&quot;&gt; &lt;length&gt;i[1].length&lt;/length&gt;
	 * <br>
	 * ... <br>
	 * &lt;/pos&gt; <br>
	 * ... <br>
	 * &lt;/endTag&gt;
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param i
	 * 			the two dimensional array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
	public static void appendInt2ArrayWithTags( StringBuffer source, int[][] i, String startTag, String endTag )
	{
		int l = i.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			help2 = IntArrayWithTags( i[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}

	/**
	 * Appends an encoded two dimensional double array with equal tags to the StringBuffer <code>source</code>.
	 * The encoding is like in appendInt2ArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param d
	 * 			the two dimensional array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded two dimensional array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendInt2ArrayWithTags( StringBuffer source, int[][] i,
	 *      String startTag, String endTag )
	 */
	public static void appendDouble2ArrayWithTags( StringBuffer source, double[][] d, String tag )
	{
		appendDouble2ArrayWithTags( source, d, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional double array with start and end tag to the StringBuffer <code>source</code>.
	 * The encoding is like in appendInt2ArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param d
	 * 			the two dimensional array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * 
	 * @see XMLParser#appendInt2ArrayWithTags( StringBuffer source, int[][] i,
	 *      String startTag, String endTag )
	 */
	public static void appendDouble2ArrayWithTags( StringBuffer source, double[][] d, String startTag, String endTag )
	{
		int l = d.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			help2 = DoubleArrayWithTags( d[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}
    
	/**
	 * Appends an encoded two dimensional {@link Storable} array with equal tags to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the two dimensional Storable array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded two dimensional Storable array should be framed (start tag should be equal to end tag)
	 * 
	 */
    public static void appendStorable2ArrayWithTags( StringBuffer source, Storable[][] s, String tag){
        appendStorable2ArrayWithTags(source,s,tag,tag);
    }
    
    /**
	 * Appends an encoded two dimensional {@link Storable} array with start and end tag to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the two dimensional Storable array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
    public static void appendStorable2ArrayWithTags( StringBuffer source, Storable[][] s, String startTag, String endTag ){
        int l = s.length;
        StringBuffer help1 = new StringBuffer( 2500 ), help2;
        help1.append( "\t" );
        appendIntWithTags( help1, l, "length" );
        for( int j = 0; j < l; j++ )
        {
            help2 = StorableArrayWithTags( s[j] );
            help2.append( "\t" );
            addTags( help2, "pos val=\"" + j + "\"", "pos" );
            help2.insert( 0, "\n\t" );
            help1.append( help2 );
        }
        addTags( help1, startTag, endTag );
        source.append( help1 );
    }
    
    /**
	 * Appends an encoded three dimensional {@link Storable} array with equal tags to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the three dimensional Storable array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded three dimensional Storable array should be framed (start tag should be equal to end tag)
	 */
    public static void appendStorable3ArrayWithTags( StringBuffer source, Storable[][][] s, String tag){
        appendStorable3ArrayWithTags(source,s,tag,tag);
    }
    
    /**
	 * Appends an encoded three dimensional {@link Storable} array with start and end tag to the StringBuffer <code>source</code>.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the three dimensional Storable array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 */
    public static void appendStorable3ArrayWithTags( StringBuffer source, Storable[][][] s, String startTag, String endTag ){
        int l = s.length;
        StringBuffer help1 = new StringBuffer( 2500 );
        help1.append( "\t" );
        appendIntWithTags( help1, l, "length" );
        for( int j = 0; j < l; j++ )
        {
            appendStorable2ArrayWithTags( help1, s[j], "pos val=\"" + j + "\"", "pos" );
        }
        addTags( help1, startTag, endTag );
        source.append( help1 );
    }

    /**
	 * Appends an encoded two dimensional String array with equal tags to the StringBuffer <code>source</code>.
	 * The encoding is like in appendInt2ArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the two dimensional array that should be encoded and appended to the source framed by the tags
	 * @param tag
	 * 			the tags by which the encoded two dimensional array should be framed (start tag should be equal to end tag)
	 * 
	 * @see XMLParser#appendInt2ArrayWithTags( StringBuffer source, int[][] i,
	 *      String startTag, String endTag )
	 */
	public static void appendString2ArrayWithTags( StringBuffer source, String[][] s, String tag )
	{
		appendString2ArrayWithTags( source, s, tag, tag );
	}

	/**
	 * Appends an encoded two dimensional String array with start and end tag to the StringBuffer <code>source</code>.
	 * The encoding is like in appendInt2ArrayWithTags.
	 * 
	 * @param source
	 * 			the source StringBuffer that should be encoded in XML
	 * @param s
	 * 			the two dimensional array that should be encoded and appended to the source framed by start and end tag
	 * @param startTag
	 * 			the start tag
	 * @param endTag
	 * 			the end tag
	 * 
	 * @see XMLParser#appendInt2ArrayWithTags( StringBuffer source, int[][] i,
	 *      String startTag, String endTag )
	 */
	public static void appendString2ArrayWithTags( StringBuffer source, String[][] s, String startTag, String endTag )
	{
		int l = s.length;
		StringBuffer help1 = new StringBuffer( 2500 ), help2;
		help1.append( "\t" );
		appendIntWithTags( help1, l, "length" );
		for( int j = 0; j < l; j++ )
		{
			help2 = StringArrayWithTags( s[j] );
			help2.append( "\t" );
			addTags( help2, "pos val=\"" + j + "\"", "pos" );
			help2.insert( 0, "\n\t" );
			help1.append( help2 );
		}
		addTags( help1, startTag, endTag );
		source.append( help1 );
	}
}