package de.jstacs.motifDiscovery;

import de.jstacs.Storable;
import de.jstacs.data.Sequence;
import de.jstacs.data.sequences.annotation.StrandedLocatedSequenceAnnotationWithLength.Strand;

/**
 * This is the interface that any tool for de novo motif discovery should implement. 
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public interface MotifDiscoverer extends Cloneable, Storable {
	
	/**
	 * This enum can be used to determine which kind of profile should be returned. 
	 *  
	 * @see MotifDiscoverer#getProfileOfScoresFor(int, int, Sequence, int, KindOfProfile)
	 */
	public enum KindOfProfile {
		/**
		 * The profile should return \log Q(seq,pos,component|class). 
		 */
		UNNORMALIZED_JOINT,
		/**
		 * The profile should return \log P(seq,pos|component,class).
		 */
		NORMALIZED_CONDITIONAL,
		/**
		 * The profile should return \log Q(seq,pos|component,class).
		 */
		UNNORMALIZED_CONDITIONAL
	}
	
	/**
	 * This method returns the length of the motif with index <code>motif</code>.
	 * 
	 * @param motif the index of the motif
	 * 
	 * @return the length of the motif with index <code>motif</code>
	 */
	public int getMotifLength( int motif );
	
	/**
	 * Returns the number of components in this <code>MotifDiscoverer</code>.
	 * 
	 * @return the number of components
	 */
	public int getNumberOfComponents();
	
	/**
	 * Returns the number of motifs for this <code>MotifDiscoverer</code>
	 * 
	 * @return the number of motifs
	 */
	public int getNumberOfMotifs();
	
	/**
	 * Returns the number of motifs that are used in the component <code>component</code> of this motif discoverer.
	 * 
	 * @param component the component
	 * 
	 * @return the number of motifs
	 */
	public int getNumberOfMotifsInComponent( int component );
	
	/**
	 * Returns the index of the component with the maximal score for the sequence <code>sequence</code>.
	 * 
	 * @param sequence the sequence
	 * 
	 * @return the index of the maximal component
	 * 
	 * @throws Exception if the index could not be computed for any reasons
	 */
	public int getIndexOfMaximalComponentFor( Sequence sequence ) throws Exception;
	
	/**
	 * Returns the global index of the <code>motif</code> used in <code>component</code>.
	 * The index returned must be at least 0 and less than <code>getNumberOfMotifs()</code>.
	 * 
	 * @param component the component index
	 * @param motif the motif index in the component
	 * 
	 * @return the global index of the <code>motif<code> in <code>component</code>
	 */
	public int getGlobalIndexOfMotifInComponent( int component, int motif );
	
	/**
	 * Returns the profile of the scores for component <code>component</code> and motif <code>motif</code> at all possible 
	 * start-positions of the motif in the sequence <code>sequence</code>. This array should be of length <code>sequence.length() - startpos - motifs[motif].length() + 1</code>.
	 *
	 * <br>
	 * 
	 * A high score should encode for a probable start position.
	 * 
	 * @param component the component
	 * @param motif the index of the motif in the component
	 * @param sequence the sequence
	 * @param startpos the start position
	 * @param kind indicates the kind of profile
	 * 
	 * @return the profile of scores
	 * 
	 * @throws Exception if the score could not be computed for any reasons
	 */
	public double[] getProfileOfScoresFor( int component, int motif, Sequence sequence, int startpos, KindOfProfile kind) throws Exception;
	
	/**
	 * This method returns the strand for a given subsequence if it is consider as site of the motif model in a specific component.
	 * 
	 * @param component the component
	 * @param motif the index of the motif in the component
	 * @param sequence the sequence
	 * @param startpos the start position in the sequence
	 * 
	 * @return the predicted strand annotation
	 * 
	 * @throws Exception if the strand could not be computed for any reasons
	 */
	public Strand getStrandFor( int component, int motif, Sequence sequence, int startpos ) throws Exception;
}
