/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.parameters;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.SimpleParameter.IllegalValueException;

/**
 * Class for a <code>Parameter</code> that contains a <code>ParameterSet</code> as value. This <code>ParameterSet</code>
 * can be set either using the constructor or using the <code>setValue()</code>-method and can be obtained
 * by the <code>getValue()</code>-method.
 * <code>ParameterSetContainer</code>s can be used to build tree-structures of <code>Parameter</code>.
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public class ParameterSetContainer extends Parameter implements Rangeable, RangeIterator{

	private String name;
	private String comment;
	private ParameterSet parameters;
    /**
     * The message of the last error or <code>null</code> if no error occurred
     */
	protected String errorMessage;
	
    /**
     * Creates an new <code>ParameterSetContainer</code> out of a <code>ParameterSet</code>.
     * @param name the name of the <code>ParameterSetContainer</code>
     * @param comment the comment of the <code>ParameterSetContainer</code>
     * @param content the <code>ParameterSet</code>
     */
	public ParameterSetContainer(String name, String comment, ParameterSet content){
		this.name = name;
		this.comment = comment;
		this.parameters = content;
		this.parameters.setParent( this );
		this.errorMessage = null;
	}
	
    /**
     * Creates a new <code>ParameterSetContainer</code> from its XML-representation.
     * @param representation the XML-representation
     * @throws NonParsableException is thrown if <code>representation</code> could not be parsed
     */
	public ParameterSetContainer(StringBuffer representation) throws NonParsableException{
		fromXML(representation);
	}
	
	public ParameterSetContainer clone() throws CloneNotSupportedException{
		ParameterSetContainer clone = (ParameterSetContainer) super.clone();
		clone.parameters = parameters == null ? null : parameters.clone();
		if(clone.parameters != null){
			clone.parameters.setParent( clone );
		}
		return clone;
	}
	
	public String getName() {
		return name;
	}

	public DataType getDatatype() {
		return DataType.PARAMETERSET;
	}

	public String getComment() {
		return comment;
	}

	public boolean isRequired() {
		return true;
	}

	public boolean checkValue(Object value) {
		return (value == null || value instanceof ParameterSet);
	}

	public void setValue(Object value) throws IllegalValueException {
		if(checkValue(value)){
			this.parameters = (ParameterSet) value;
			this.parameters.setParent( this );
		}else{
			throw new IllegalValueException("Only parameter sets allowed for ParameterSetContainer!");
		}
	}

	public ParameterSet getValue() {
		return parameters;
	}

	public boolean hasDefaultOrIsSet() {
		return parameters != null
			&& parameters.hasDefaultOrIsSet();
	}

	public boolean isSet() {
		return parameters != null;
	}

	public boolean isAtomic() {
		return false;
	}

	public String getErrorMessage() {
		return parameters.getErrorMessage();
	}

	public void simplify() {
		parameters.simplify();
	}

	public void reset() {
		parameters.reset();
	}

	public void setDefault(Object defaultValue) throws Exception {
		throw new Exception("Not applicable to ParameterSetContainer");
	}

	public StringBuffer toXML() {
		StringBuffer buf = super.toXML();
		XMLParser.addTags( buf, "superParameter" );
		XMLParser.appendStringWithTags(buf,name,"name");
		XMLParser.appendStringWithTags(buf,comment,"comment");
		XMLParser.appendStringWithTags(buf,errorMessage,"errorMessage");
		XMLParser.appendStorableOrNullWithTags(buf,parameters,"parameters");
		XMLParser.addTags(buf,"parameterSetContainer");
		return buf;
	}

	protected void fromXML(StringBuffer representation)
			throws NonParsableException {
		representation = XMLParser.extractForTag(representation, "parameterSetContainer");
		super.fromXML( XMLParser.extractForTag( representation, "superParameter" ) );
		name = XMLParser.extractStringForTag(representation,"name");
		comment = XMLParser.extractStringForTag(representation,"comment");
		errorMessage = XMLParser.extractStringForTag(representation, "errorMessage");
        if(errorMessage != null && errorMessage.equals("null")){
            errorMessage = null;
        }
		parameters = (ParameterSet) XMLParser.extractStorableOrNullForTag(representation,"parameters");
		parameters.setParent( this );
	}

    public boolean isRangeable() {
        return true;
    }

    public Parameter getRangedInstance() throws Exception {
        this.parameters.makeRanged();
        return this;
    }

    public boolean next() throws ParameterException {
        return parameters.next();
    }

    public void resetToFirst() {
    	parameters.resetToFirst();
    }
    
    public boolean isRanged(){
        return parameters.isRanged();
    }

    public int getNumberOfValues() {
        return parameters.getNumberOfValues();
    }

    public String valuesToString() {
        //return name;
    	return "(" + parameters.valuesToString() + ")";
    }

}
