/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.parameters.validation;

import de.jstacs.NonParsableException;
import de.jstacs.io.XMLParser;
import de.jstacs.parameters.Parameter;

/**
 * Class for a <code>Constraint</code> that defines a condition on one <code>Parameter</code> (the one containing
 * the <code>ConstraintValidator</code>) with respect to another <code>Parameter</code>.
 * @author Jan Grau
 *
 */
public abstract class ReferenceConstraint implements Constraint {

    /**
     * The reference to the <code>Parameter</code> that is part of the condition
     */
	protected Parameter constraintParameter;
    /**
     * The message of the last error or <code>null</code>
     */
	protected String errorMessage; 
	
    /**
     * Creates a new <code>ReferenceCoinstraint</code> with respect to the <code>Parameter</code> <code>constraintParameter</code>.
     * @param constraintParameter the parameter
     */
	public ReferenceConstraint(Parameter constraintParameter){
		this.constraintParameter = constraintParameter;
	}
	
    /**
     * Creates a new <code>ReferenceConstraint</code> from its XML-representation.
     * @param representation the XML-representation
     * @throws NonParsableException is thrown if <code>representation</code> could not be parsed
     */
	public ReferenceConstraint(StringBuffer representation) throws NonParsableException{
		fromXML(representation);
	}
	
	public abstract ReferenceConstraint clone() throws CloneNotSupportedException;
	
    
    /**
     * Sets the standard-fields of <code>clone</code> to the values (or clones of them) of this instance.
     * @param clone the clone
     * @throws CloneNotSupportedException is thrown if the contained <code>Parameter</code> could not be cloned.
     */
	protected void fillWithStandardFieldForClone(ReferenceConstraint clone) throws CloneNotSupportedException{
		clone.constraintParameter = constraintParameter.clone();
		clone.errorMessage = errorMessage;
	}
	
	public abstract boolean check(Object value);
	
	public String getErrorMessage(){
		return errorMessage;
	}
	
	public StringBuffer toXML(){
		StringBuffer buf = new StringBuffer();
		XMLParser.appendStorableWithTags(buf,constraintParameter,"parameter");
		XMLParser.appendStringWithTags(buf,errorMessage,"errorMessage");
		XMLParser.addTags(buf,"referenceConstraint");
		return buf;
	}
	
	/**
	 * Parses a {@link ReferenceConstraint} from the XML-representation
	 * as returned by {@link ReferenceConstraint#toXML()}.
	 * @param representation the XML-representation
	 * @throws NonParsableException thrown if the XML-code could not be parsed
	 */
	public void fromXML(StringBuffer representation) throws NonParsableException{
		representation = XMLParser.extractForTag(representation,"referenceConstraint");
		constraintParameter = (Parameter) XMLParser.extractStorableForTag(representation,"parameter");
		errorMessage = XMLParser.extractStringForTag(representation,"errorMessage");
	}

}
