/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.results;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.Storable;
import de.jstacs.classifier.AbstractClassifier;
import de.jstacs.io.XMLParser;
import de.jstacs.models.AbstractModel;
import de.jstacs.models.Model;

/**
 * Class for results that are {@link Storable}s. The {@link Storable#toXML()} method is used
 * to save the {@link StorableResult} together with the result to an XML-representation, or in the {@link StorableResult#toString()} method.
 * 
 * @see de.jstacs.Storable
 * @author Jan Grau
 */
public class StorableResult extends Result {
	
	/**
	 * the {@link Storable} cannot be trained anyway.
	 */  
    public static final byte NA = -1;
    /**
     * The model/classifier has been trained
     */
    public static final byte FALSE = 0;
    /**
     * the model/classifier has been trained
     */
    public static final byte TRUE = 1;
	
	/**
     * The {@link Storable} that is the result
	 */
	private Storable object;
    
	/**
	 * Creates a result for an XML-representation of an object.
	 * @param name the name of the result
	 * @param comment a comment on the result
	 * @param object the {@link Storable} that is the result
	 */
	public StorableResult(String name, String comment, Storable object){
		super( name, comment, DataType.STORABLE );
		this.object = object;		
	}
	
	/**
	 * Constructs a new <code>ObjectResult</code> from its XML-representation
	 * @param buf the XML-representation
	 * @throws NonParsableException a <code>NonParsableException</code> is thrown if <code>buf</code> could not be parsed
	 */
	public StorableResult(StringBuffer buf) throws NonParsableException{
		super(buf);
	}
	
	/**
	 * Returns <code>TRUE</code> if the model or classifier was trained when obtaining its XML-representation
	 * stored in this <code>ObjectResult</code>, <code>FALSE</code> if it was not, and <code>NA</code>
     * if the object could not be trained anyway.
	 * @return if the model or classifier was trained
	 */
	public byte isTrained(){
        if(object instanceof AbstractModel){
            return ( (Model) object).isTrained() ? TRUE : FALSE;
        }else if(object instanceof AbstractClassifier){
            return ( (AbstractClassifier) object).isTrained() ? TRUE : FALSE;
        }else{
            return NA;
        }
	}
	
	/**
	 * Returns the name of the class of the <code>Storable</code> corresponding to the XML-representation
	 * stored in this <code>ObjectResult</code>.
	 * @return the name of the class
	 */
	public String getClassName(){
		return object.getClass().getName();
	}

	public String getResult() {
		return object.toXML().toString();
	}

    /**
     * Returns the instance of the <code>Storable</code> that is the result of
     * this <code>ObjectResult</code>
     * @return the instance
     */
    public Storable getResultInstance(){
        return object;
    }

	public StringBuffer toXML() {
		StringBuffer buf = new StringBuffer();
		appendMainInfo( buf );
		XMLParser.appendStorableWithTags(buf,object,"object");
		XMLParser.addTags(buf,"objectResult");
		return buf;
	}

	protected void fromXML(StringBuffer representation) throws NonParsableException {
		representation = XMLParser.extractForTag(representation,"objectResult");
		extractMainInfo( representation );
		this.object = XMLParser.extractStorableForTag(representation,"object");
	}
    
    public String toString(){
        return name+":\n"+object.toXML()+"\n";
    }
}
