/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.scoringFunctions.homogeneous;

import java.util.Arrays;

import de.jstacs.NonParsableException;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sample;
import de.jstacs.data.Sequence;
import de.jstacs.io.XMLParser;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;

/**
 * This scoring function does nothing. So it is possible to save parameters in an optimization.
 * 
 * @author Jens Keilwagen, Jan Grau
 */
public class UniformHomogeneousScoringFunction extends HomogeneousScoringFunction
{
	private double ess, p, logP;
	
	/**
	 * This is the main constructor that creates an instance that models each sequence uniformly.
	 * 
	 * @param alphabets the AlphabetContainer
	 * @param ess the equivalent sample size (ess) for the class
	 */
	public UniformHomogeneousScoringFunction( AlphabetContainer alphabets, double ess )
	{
		super( alphabets );
		if( ess < 0 )
		{
			throw new IllegalArgumentException( "The given ess has to be non-negative." );
		}
		this.ess = ess;
		computeLogP();
	}
	
	private void computeLogP()
	{
		p = 1d / alphabets.getAlphabetLengthAt(0);
		logP = Math.log(p);
	}

	/**
	 * This is the constructor for {@link de.jstacs.Storable}.
	 * 
	 * @param xml the xml representation
	 * 
	 * @throws NonParsableException if the representation could not be parsed.
	 */
	public UniformHomogeneousScoringFunction( StringBuffer xml ) throws NonParsableException
	{
		super( xml );
	}

	public String getInstanceName()
	{
		return "uniform";
	}

	public double getLogScore( Sequence seq, int start, int length )
	{
		return length*logP;
	}

	public double getLogScoreAndPartialDerivation( Sequence seq, int start, int length, IntList indices, DoubleList dList )
	{
		return length*logP;
	}

	public int getNumberOfParameters()
	{
		return 0;
	}

	public void setParameters( double[] params, int start )
	{
	}

	public StringBuffer toXML()
	{
		StringBuffer b = new StringBuffer(1000);
		XMLParser.appendIntWithTags( b, length, "length" );
		XMLParser.appendStorableWithTags( b, alphabets, "alphabets" );
		XMLParser.appendDoubleWithTags( b, ess, "ess" );
		XMLParser.addTags( b, getClass().getSimpleName() );
		return b;		
	}

	public double getNormalizationConstant( int length )
	{
		return 1;
	}

	public void initializeFunction( int index, boolean meila, Sample[] data, double[][] weights )
	{
		// does nothing
	}

	protected void fromXML( StringBuffer xml ) throws NonParsableException
	{
		StringBuffer b = XMLParser.extractForTag( xml, getClass().getSimpleName() );
		length = XMLParser.extractIntForTag( b, "length" );
		alphabets = (AlphabetContainer) XMLParser.extractStorableForTag( b, "alphabets" );
		ess = XMLParser.extractDoubleForTag( b, "ess" );
		computeLogP();
	}

	public int getSizeOfEventSpaceForRandomVariablesOfParameter(int index) 
	{
		return 0;
	}

	public double getPartialNormalizationConstant( int parameterIndex, int length ) throws Exception
	{
		throw new IndexOutOfBoundsException( "Since a uniform scoring function has no parameters, this method can not be used" );
	}

	public double getEss()
	{
		return ess;
	}
	
	public String toString()
	{
		return p + " for each element of " + alphabets.getAlphabetAt(0).toString();
	}

	public double getLogPriorTerm()
	{
		return 0;
	}

	public void addGradientOfLogPriorTerm( double[] grad, int start ){}

	public double[] getCurrentParameterValues() throws Exception
	{
		return new double[0];
	}

	public boolean isInitialized()
	{
		return true;
	}
	
	public boolean isNormalized()
	{
		return true;
	}

	public int getMaximalMarkovOrder()
	{
		return 0;
	}
	
	public void initializeFunctionRandomly(boolean freeParams) throws Exception
	{	
	}
	
	public double[][][] getAllConditionalStationaryDistributions()
	{
		double[][][] res = new double[1][1][(int)alphabets.getAlphabetLengthAt(0)];
		Arrays.fill( res[0][0], p );
		return res;
	}

	public void setStatisticForHyperparameters( int[] length, double[] weights ) throws Exception
	{
		//does nothing
	}
}
