/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.classifier.scoringFunctionBased.cll;

import de.jstacs.DataType;
import de.jstacs.NonParsableException;
import de.jstacs.classifier.scoringFunctionBased.ScoreClassifier;
import de.jstacs.classifier.scoringFunctionBased.ScoreClassifierParameterSet;
import de.jstacs.classifier.scoringFunctionBased.OptimizableFunction.KindOfParameter;
import de.jstacs.data.AlphabetContainer;
import de.jstacs.parameters.SimpleParameter;

/**
 * This class contains the parameters for the {@link CLLClassifier}.
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public class CLLClassifierParameterSet extends ScoreClassifierParameterSet {

	/**
	 * The standard constructor for the interface {@link de.jstacs.Storable}.
	 * Constructs a {@link CLLClassifierParameterSet} out of its XML
	 * representation.
	 * 
	 * @param xml
	 *            the XML representation as {@link StringBuffer}
	 * 
	 * @throws NonParsableException
	 *             if the {@link CLLClassifierParameterSet} could not be
	 *             reconstructed out of the XML representation (the
	 *             {@link StringBuffer} could not be parsed)
	 * 
	 * @see ScoreClassifierParameterSet#ScoreClassifierParameterSet(StringBuffer)
	 * @see de.jstacs.Storable
	 */
	public CLLClassifierParameterSet( StringBuffer xml ) throws NonParsableException {
		super( xml );
	}

	/**
	 * The default constructor that constructs a new
	 * {@link CLLClassifierParameterSet}.
	 * 
	 * @param alphabet
	 *            the {@link AlphabetContainer}
	 * @param length
	 *            the length of the sequences
	 * @param algo
	 *            the algorithm that shall be used for optimization
	 * @param eps
	 *            the epsilon for stopping the optimization
	 * @param lineps
	 *            the epsilon for stopping the line search
	 * @param startD
	 *            the start distance for the line search
	 * @param free
	 *            the switch for using only the free or all parameters in a
	 *            {@link de.jstacs.scoringFunctions.ScoringFunction}
	 * @param kind
	 *            indicates the kind of class parameter initialization
	 * @param norm
	 *            the switch for using a normalization while optimization
	 * 
	 * @throws Exception
	 *             if something went wrong
	 * 
	 * @see KindOfParameter
	 * @see CLLClassifierParameterSet#CLLClassifierParameterSet(Class,
	 *      de.jstacs.data.AlphabetContainer, int, byte, double, double, double, boolean,
	 *      de.jstacs.classifier.scoringFunctionBased.OptimizableFunction.KindOfParameter, boolean)
	 */
	public CLLClassifierParameterSet( AlphabetContainer alphabet, int length, byte algo, double eps, double lineps, double startD,
										boolean free, KindOfParameter kind, boolean norm ) throws Exception {
		this( CLLClassifier.class, alphabet, length, algo, eps, lineps, startD, free, kind, norm );
	}

	/**
	 * The default constructor that constructs a new
	 * {@link CLLClassifierParameterSet}.
	 * 
	 * @param instanceClass
	 *            the class of the instance
	 * @param alphabet
	 *            the {@link AlphabetContainer}
	 * @param length
	 *            the length of the sequences
	 * @param algo
	 *            the algorithm that shall be used for optimization
	 * @param eps
	 *            the epsilon for stopping the optimization
	 * @param lineps
	 *            the epsilon for stopping the line search
	 * @param startD
	 *            the start distance for the line search
	 * @param free
	 *            the switch for using only the free or all parameters in a
	 *            {@link de.jstacs.scoringFunctions.ScoringFunction}
	 * @param kind
	 *            indicates the kind of class parameter initialization
	 * @param norm
	 *            the switch for using a normalization while optimization
	 * 
	 * @throws Exception
	 *             if something went wrong
	 * 
	 * @see KindOfParameter
	 * @see ScoreClassifierParameterSet#ScoreClassifierParameterSet(Class,
	 *      de.jstacs.data.AlphabetContainer, int, byte, double, double, double, boolean,
	 *      de.jstacs.classifier.scoringFunctionBased.OptimizableFunction.KindOfParameter)
	 */
	protected CLLClassifierParameterSet( Class<? extends ScoreClassifier> instanceClass, AlphabetContainer alphabet, int length, byte algo,
											double eps, double lineps, double startD, boolean free, KindOfParameter kind, boolean norm )
																																		throws Exception {
		super( instanceClass, alphabet, length, algo, eps, lineps, startD, free, kind );
		parameters.get( 6 ).setValue( norm );
	}

	/* (non-Javadoc)
	 * @see de.jstacs.classifier.scoringFunctionBased.ScoreClassifierParameterSet#loadParameters()
	 */
	@Override
	protected void loadParameters() throws Exception {
		super.loadParameters();
		parameters.add( new SimpleParameter( DataType.BOOLEAN,
				"Normalize",
				"If true the conditional likelihood will be normalized to the number of samples.",
				true,
				new Boolean( true ) ) );
	}

	/**
	 * This method indicates if a normalization shall be used while
	 * optimization. The normalization is done through division by the number of
	 * sequences.
	 * 
	 * @return <code>true</code> if a normalization shall be used while
	 *         optimization, <code>false</code> otherwise
	 */
	public boolean shouldBeNormalized() {
		return (Boolean)getParameterAt( 6 ).getValue();
	}
}
