package de.jstacs.motifDiscovery;

/**
 * This interface allows to modify a motif model.
 * 
 * @author Jan Grau, Jens Keilwagen
 */
public interface Mutable {

	/**
	 * This method determines the number of not significant positions from each
	 * side of the motif using the the significance level <code>sign</code>
	 * and the contrast distributions of the left or right side,
	 * <code>contrastLeft</code> and <code>contrastRight</code>,
	 * respectively. The contrast array have four dimensions.
	 * <ol>
	 * <li>The first is used for the possibility of having different contrast (caused by different flanking models).
	 * <li>The second is used for the order of the contrast.
	 * <li>The third is used for the (encoded) context.
	 * <li>The fourth is used for the the realization of the random variable
	 * </ol> 
	 * For example, if we have only one flanking model which is a homogeneous Markov model of order 0 for a {@link de.jstacs.data.alphabets.DNAAlphabet},
	 * the contrast array has the dimension <code>new double[1][1][1][4]</code>. For the same example but with order 1, the contrast array has the dimension
	 * <code>new double[1][1][4][4]</code>. Left and right contrast can have different dimensions.
	 * 
	 * @param samples the summed weights of {@link de.jstacs.data.Sequence} containing this motif
	 * @param weightsLeft the weights for the left contrast distributions 
	 * @param weightsRight the weights for the right contrast distributions
	 * @param contrastLeft
	 *            the left contrast distributions
	 * @param contrastRight
	 *            the right contrast distributions
	 * @param sign
	 *            the significance level
	 * 
	 * @return a two dimensional array containing at position 0 the number of
	 *         not significant positions from the left side using
	 *         <code>contrastLeft</code> and at position 1 the number of not
	 *         significant positions from the right side using
	 *         <code>contrastRight</code>
	 *         
	 * @see Mutable#modify(double[], double[], double[][][][], double[][][][], int, int)
	 */
	public int[] determineNotSignificantPositions(double samples, double[] weightsLeft, double[] weightsRight,
			double[][][][] contrastLeft, double[][][][] contrastRight, double sign);

	/**
	 * Manually modifies the model. The two offsets <code>offsetLeft</code>
	 * and <code>offsetRight</code> define how many positions the left or
	 * right border positions shall be moved. Negative numbers indicate moves to
	 * the left while positive numbers correspond to moves to the right.
	 * 
	 * @param weightsLeft the weights for the left replacement distributions 
	 * @param weightsRight the weights for the left replacement distributions
	 * @param replacementLeft
	 *            the replacement distribution for the left side
	 * @param replacementRight
	 *            the replacement distribution for the right side
	 * @param offsetLeft
	 *            the offset on the left side
	 * @param offsetRight
	 *            the offset on the right side
	 * 
	 * @return <code>true</code> if the motif model was modified otherwise
	 *         <code>false</code>
	 */
	public boolean modify(double[] weightsLeft, double[] weightsRight, double[][][][] replacementLeft,
			double[][][][] replacementRight, int offsetLeft, int offsetRight);
}
