package de.jstacs.motifDiscovery;

import de.jstacs.data.Sample;

/**
 * This is the interface that any tool for de-novo motif discovery should implement that allows any modify-operations like shift, shrink and expand.
 * These operations are possible if the motif is mutable.  
 * 
 * @author Jan Grau, Jens Keilwagen
 * 
 * @see de.jstacs.motifDiscovery.Mutable
 */
public interface MutableMotifDiscoverer extends MotifDiscoverer {

	/**
	 * This method determines the number of not significant positions from each
	 * side of the motif with index <code>motif</code>.
	 * 
	 * @param motif the index of the motif in the motif discoverer
	 * @param data an array {@link Sample}s, each array-entry represents on class 
	 * @param weights the weights of each {@link de.jstacs.data.Sequence} for each class
	 * @param classIdx the index of the current class in <code>classCounts</code>
	 * 
	 * @return a two dimensional array containing at position 0 the number of
	 *         not significant positions from the left and at position 1 the
	 *         number of not significant positions from the right side
	 * 
	 * @see Mutable#determineNotSignificantPositions(double, double[], double[], double[][][][], double[][][][], double)
	 * @see MutableMotifDiscoverer#modifyMotif(int, int, int)
	 * @see MutableMotifDiscoverer#modifyMotif(int, double[], double[], double[][][][], double[][][][], int, int)
	 */
	public int[] determineNotSignificantPositionsFor( int motif, Sample[] data, double[][] weights, int classIdx );
	
	/**
	 * Manually modifies the motif model with index <code>motifIndex</code>. The two offsets <code>offsetLeft</code> and <code>offsetRight</code>
	 * define how many positions the left or right border positions shall be moved. Negative numbers indicate moves to the left while positive
	 * numbers correspond to moves to the right.
	 * 
	 * @param motifIndex the index of the motif in the motif discoverer
	 * @param weightsLeft the weights for the left contrast distributions 
	 * @param weightsRight the weights for the right contrast distributions
	 * @param replacementLeft the replacement distribution for the left side
	 * @param replacementRight the replacement distribution for the right side
	 * @param offsetLeft the offset on the left side
	 * @param offsetRight the offset on the right side
	 * 
	 * @return <code>true</code> if the motif model was modified otherwise <code>false</code>
	 * 
	 * @throws Exception 
	 */
	public boolean modifyMotif( int motifIndex, double[] weightsLeft, double[] weightsRight, double[][][][] replacementLeft, double[][][][] replacementRight, int offsetLeft, int offsetRight ) throws Exception;
	
	/**
	 * Manually modifies the motif model with index <code>motifIndex</code>. The two offsets <code>offsetLeft</code> and <code>offsetRight</code>
	 * define how many positions the left or right border positions shall be moved. Negative numbers indicate moves to the left while positive
	 * numbers correspond to moves to the right. The distribution for sequences to the left and right side of the motif shall be computed internally.
	 * 
	 * @param motifIndex the index of the motif in the motif discoverer
	 * @param offsetLeft the offset on the left side
	 * @param offsetRight the offset on the right side
	 * 
	 * @return <code>true</code> if the motif model was modified otherwise <code>false</code>
	 * 
	 * @throws Exception 
	 * 
	 * @see MutableMotifDiscoverer#modifyMotif(int, double[], double[], double[][][][], double[][][][], int, int)
	 */
	public boolean modifyMotif( int motifIndex, int offsetLeft, int offsetRight ) throws Exception;
	
	/**
	 * This method allows to initialize the model of a motif manually using a weighted sample.
	 * 
	 * @param motifIndex the index of the motif in the motif discoverer
	 * @param data the sample of sequences
	 * @param weights either <code>null</code> or an array of length <code>data.getNumberofElements()</code> with non-negative weights.
	 * 
	 * @throws Exception if initialize was not possible
	 */
	public void initializeMotif( int motifIndex, Sample data, double[] weights ) throws Exception;
}
