package de.jstacs.scoringFunctions.mix.motifSearch;

import de.jstacs.NonParsableException;
import de.jstacs.data.Sample;
import de.jstacs.utils.DoubleList;
import de.jstacs.utils.IntList;

/**
 * This scoring function implements a uniform distribution for positions.
 * The class has no parameters, so the distribution does not change and it is possible to save parameters in an optimization.
 */
public final class UniformDurationScoringFunction extends DurationScoringFunction
{
	private double logP;
	
	/**
	 * This is the main constructor that creates an instance for the given interval.
	 * 
	 * @param min the minimal value
	 * @param max the maximal value
	 */
	public UniformDurationScoringFunction( int min, int max )
	{
		this( min, max, 0 );
	}
	
	/**
	 * This is the main constructor that creates an instance for the given interval and given ESS.
	 * 
	 * @param min the minimal value
	 * @param max the maximal value
	 * @param ess the equivalent sample size (used for the class probability)
	 */
	public UniformDurationScoringFunction( int min, int max, double ess )
	{
		super( min, max, ess );
		computeLogP();
	}

	/**
	 * This is the constructor for {@link de.jstacs.Storable}. Creates a new
	 * {@link UniformDurationScoringFunction} out of a {@link StringBuffer}.
	 * 
	 * @param b
	 *            the XML representation as {@link StringBuffer}
	 * 
	 * @throws NonParsableException
	 *             if the XML representation could not be parsed
	 */
	public UniformDurationScoringFunction( StringBuffer b ) throws NonParsableException
	{
		super( b );
		computeLogP();
	}
	
	private void computeLogP()
	{
		logP = -Math.log( delta + 1 );
	}

	public String getInstanceName()
	{
		return "uniform";
	}

	public int getNumberOfParameters()
	{
		return 0;
	}

	public void setParameters( double[] params, int start )
	{
	}

	public void initializeFunction( int index, boolean meila, Sample[] data, double[][] weights )
	{
		// does nothing
	}
	
	protected String getRNotation( String distributionName )
	{
		return "l = " + min + ":" + max + "; n = length(l); " + distributionName + " = rep(1,n)/n;";
	}
	
	public double getLogPriorTerm()
	{
		// since the normalization constant does not depend on any parameter,
		// it is constant and therefore left out
		return 0;
	}

	public void addGradientOfLogPriorTerm( double[] grad, int start ){}

	public double getLogScore( int... values )
	{
		return logP;
	}

	protected double getLogScoreAndPartialDerivation( IntList indices, DoubleList partialDer, int... values )
	{
		return logP;
	}

	/**
	 * This method draws from the distribution and returns the result in the given array.
	 * 
	 * @param positions an array for the result.
	 */
	public void drawPosition( int[] positions )
	{
		positions[0] = min + r.nextInt( delta + 1 );		
	}

	public double[] getCurrentParameterValues() throws Exception
	{
		return new double[0];
	}

	public boolean isInitialized()
	{
		return true;
	}
	
	public boolean isNormalized()
	{
		return true;
	}
	
	public void initializeFunctionRandomly(boolean freeParams) throws Exception
	{	
	}

	public void initializeUniformly(){}

	public void adjust( int[] length, double[] weight ){}
}
