/*
 * This file is part of Jstacs.
 *
 * Jstacs is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Jstacs is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Jstacs.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */

package de.jstacs.algorithms.optimization.termination;

import de.jstacs.utils.Time;

/**
 * This class allows to use many {@link TerminationCondition}s at once.
 * 
 * @author Jens Keilwagen
 */
public class CombinedCondition implements TerminationCondition {

	private TerminationCondition[] condition;
	private int threshold;
	
	/**
	 * This constructor creates an instance that allows to use many {@link TerminationCondition}s at one. 
	 * 
	 * @param threshold the number of conditions that has to be fulfilled;
	 * 			if all conditions should be fulfilled than <code>threshold=condition.length</code> (equal to AND);
	 * 			if at least one condition should be fulfilled than <code>threshold=1</code> (equal to OR);
	 * @param condition the conditions that are used to create this instance and that are used to determine whether another iteration should be done
	 */
	public CombinedCondition( int threshold, TerminationCondition...  condition ) {
		if( threshold > condition.length ) {
			throw new IllegalArgumentException( "The threshold has to be smaller than the number of conditions" );
		}
		this.threshold = threshold;
		this.condition = condition; 
	}
		
	public boolean doNextIteration( int iteration, double f_last, double f_current, double[] gradient, double[] direction, double alpha,
			Time t ) {
		int positive = 0;
		for( int i = 0; i < condition.length; i++ ) {
			if( condition[i].doNextIteration( iteration, f_last, f_current, gradient, direction, alpha, t ) ){
				positive++;
			}
		}
		return positive >= threshold;		
	}
}
