/*
 * This file is part of Jstacs.
 * 
 * Jstacs is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * Jstacs is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * Jstacs. If not, see <http://www.gnu.org/licenses/>.
 * 
 * For more information on Jstacs, visit http://www.jstacs.de
 */
package de.jstacs.data.sequences.annotation;

import de.jstacs.data.AlphabetContainer;
import de.jstacs.data.Sequence;


/**
 * This class implements an {@link SequenceAnnotationParser} that parses a {@link ReferenceSequenceAnnotation} from the comment lines of a sequence.
 * 
 * @author Jan Grau
 */
public class ReferenceSequenceAnnotationParser extends SplitSequenceAnnotationParser {

	private String key;
	private AlphabetContainer alphabet;
	
	/**
	 * Creates a new {@link ReferenceSequenceAnnotationParser} with user-specified delimiters.
	 * 
	 * @param key the key for the {@link ReferenceSequenceAnnotation}
	 * @param alphabet the {@link AlphabetContainer} of the reference sequence
	 * @param keyValueDelimiter the delimiter between key and corresponding value
	 * @param annotationDelimiter the delimiter between different {@link SequenceAnnotation}s
	 * 
	 * @throws IllegalArgumentException if the delimiters are identical
	 */
	public ReferenceSequenceAnnotationParser( String key, AlphabetContainer alphabet, String keyValueDelimiter, String annotationDelimiter ) throws IllegalArgumentException {
		super( keyValueDelimiter, annotationDelimiter );
		this.key = key;
		this.alphabet = alphabet;
	}

	protected void add( String type, String identifier ) {
		if( type.equalsIgnoreCase( this.key ) ) {
			try{
				annot.add( new ReferenceSequenceAnnotation( key, Sequence.create( alphabet, identifier ) ) );
			}catch(Exception e){
				RuntimeException re = new RuntimeException( e.getMessage() );
				re.setStackTrace( e.getStackTrace() );
				throw re;
			}
		} else {
			super.add( type, identifier );
		}
	}
}
